<?php

/**
 * @package     Joomla.Administrator
 * @subpackage  com_easystore
 *
 * @copyright   (C) 2023 - 2024 JoomShaper. <https://www.joomshaper.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

use Joomla\CMS\Factory;
use Joomla\Registry\Registry;
use Joomla\CMS\Installer\Installer;
use Joomla\Database\DatabaseInterface;
use Joomla\CMS\Component\ComponentHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Install script class
 *
 * @since 1.0.0
 */
class com_easystoreInstallerScript
{
    /**
     * Method to uninstall the component
     *
     * @return void
     *
     * @since 1.0.0
     */
    public function uninstall($parent)
    {
        $extensions = [
            ['type' => 'module', 'name' => 'mod_easystore_cart_icon'],
            ['type' => 'plugin', 'name' => 'paypal', 'group' => 'easystore'],
            ['type' => 'plugin', 'name' => 'easystoreupdater', 'group' => 'system'],
            ['type' => 'plugin', 'name' => 'easystoremail', 'group' => 'system'],
            ['type' => 'plugin', 'name' => 'easystore', 'group' => 'finder'],
        ];

        foreach ($extensions as $key => $extension) {
            $db    = Factory::getContainer()->get(DatabaseInterface::class);
            $query = $db->getQuery(true);
            $query->select($db->quoteName(['extension_id']));
            $query->from($db->quoteName('#__extensions'));
            $query->where($db->quoteName('type') . ' = '. $db->quote($extension['type']));
            $query->where($db->quoteName('element') . ' = '. $db->quote($extension['name']));
            $db->setQuery($query);
            $id = $db->loadResult();

            if (isset($id) && $id) {
                $installer = new Installer();
                $installer->uninstall($extension['type'], $id);
            }
        }
    }

    /**
     * Method to run after an install/update/uninstall method
     *
     * @return mixed
     *
     * @since 1.0.0
     */
    public function postflight($type, $parent)
    {
        if ($type === 'uninstall') {
            return;
        }

        $extensions = [
            ['type' => 'module', 'name' => 'mod_easystore_cart_icon'],
            ['type' => 'plugin', 'name' => 'paypal', 'group' => 'easystore'],
            ['type' => 'plugin', 'name' => 'easystoreupdater', 'group' => 'system'],
            ['type' => 'plugin', 'name' => 'easystoremail', 'group' => 'system'],
            ['type' => 'plugin', 'name' => 'easystore', 'group' => 'finder'],
        ];

        foreach ($extensions as $key => $extension) {
            $ext       = $parent->getParent()->getPath('source') . '/' . $extension['type'] . 's/' . $extension['name'];
            $installer = new Installer();
            $installer->install($ext);

            if ($extension['type'] === 'plugin') {
                $db    = Factory::getContainer()->get(DatabaseInterface::class);
                $query = $db->getQuery(true);

                $fields     = [$db->quoteName('enabled') . ' = 1'];
                $conditions = [
                    $db->quoteName('type') . ' = ' . $db->quote($extension['type']),
                    $db->quoteName('element') . ' = ' . $db->quote($extension['name']),
                    $db->quoteName('folder') . ' = ' . $db->quote($extension['group']),
                ];

                $query->update($db->quoteName('#__extensions'))->set($fields)->where($conditions);
                $db->setQuery($query);
                $db->execute();
            }
        }

        $this->addInstallationDate();
        $this->insertRootCategory();
        $this->saveEmailTemplates($parent->getParent()->getPath('source'));
    }

    private function addInstallationDate()
    {
        if (!ComponentHelper::isInstalled('com_easystore')) {
            return;
        }

        $component   = ComponentHelper::getComponent('com_easystore');
        $params      = $component->getParams();
        $installedOn = $params->get('installed_on', null);

        if (!empty($installedOn)) {
            return;
        }

        $params->set('installed_on', Factory::getDate('now')->toSql());
        $db    = Factory::getDbo();
        $query = $db->getQuery(true);

        $fields = [
            $db->quoteName('params') . ' = ' . $db->quote($params->toString()),
        ];

        $conditions = [
            $db->quoteName('extension_id') . ' = ' . $component->id,
        ];

        $query->update($db->quoteName('#__extensions'))->set($fields)->where($conditions);
        $db->setQuery($query);

        try {
            $db->execute();
        } catch (Throwable $error) {
            throw $error;
        }
    }

    private function hasRootCategory()
    {
        $db    = Factory::getDbo();
        $query = $db->getQuery(true);

        $query->select('id')
            ->from($db->quoteName('#__easystore_categories'))
            ->where($db->quoteName('alias') . ' = ' . $db->quote('root'));

        $db->setQuery($query);

        try {
            $category = $db->loadResult() ?? null;

            return !empty($category);
        } catch (Throwable $error) {
            throw $error;
        }

        return false;
    }

    private function insertRootCategory()
    {
        if ($this->hasRootCategory()) {
            return;
        }

        $data = (object) [
            'id'        => null,
            'title'     => 'ROOT',
            'alias'     => 'root',
            'published' => 1,
            'created'   => Factory::getDate('now')->toSql(),
        ];

        $db = Factory::getDbo();

        try {
            $db->insertObject('#__easystore_categories', $data, 'id');
        } catch (Throwable $error) {
            throw $error;
        }
    }


    /**
     * Save the email templates on after installing the component.
     * This will check the emails json and update the settings accordingly
     */
    private static function getSettings()
    {
        $db    = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true);
        $query->select($db->quoteName(['key', 'value']))->from($db->quoteName('#__easystore_settings'));
        $db->setQuery($query);

        try {
            $settings = $db->loadObjectList('key') ?? [];

            foreach ($settings as &$setting) {
                $setting = json_decode($setting->value) ?? null;
            }

            unset($setting);

            $settings = (object) $settings;

            return new Registry($settings);
        } catch (Throwable $error) {
            return new Registry([]);
        }
    }

    public function saveEmailTemplates($base)
    {
        $settings       = static::getSettings();
        $savedTemplates = $settings->get('email_templates');

        $localTemplatePath = $base . '/media/data/emails.json';

        if (!file_exists($localTemplatePath)) {
            return;
        }

        $localTemplates = json_decode(file_get_contents($localTemplatePath));

        if (empty($savedTemplates)) {
            $this->saveToDatabase($localTemplates);
            return;
        }

        $newGroups     = $this->getNewGroups($savedTemplates, $localTemplates);
        $removedGroups = $this->getRemovedGroups($savedTemplates, $localTemplates);

        $savedTemplates = $this->removeGroups($savedTemplates, $removedGroups);
        $savedTemplates = $this->addNewGroups($savedTemplates, $newGroups);
        $savedTemplates = $this->updateGroups($savedTemplates, $localTemplates);

        $this->saveToDatabase($savedTemplates);
    }

    private function getNewGroups($saved, $local)
    {
        $savedGroups = array_keys((array) $saved);
        $localGroups = array_keys((array) $local);

        return array_values(array_diff($localGroups, $savedGroups));
    }

    private function getRemovedGroups($saved, $local)
    {
        $savedGroups = array_keys((array) $saved);
        $localGroups = array_keys((array) $local);

        return array_values(array_diff($savedGroups, $localGroups));
    }

    private function removeGroups($data, $groups)
    {
        if (!empty($groups)) {
            foreach ($groups as $group) {
                unset($data->$group);
            }
        }

        return $data;
    }

    private function addNewGroups($data, $groups)
    {
        if (!empty($groups)) {
            foreach ($groups as $group) {
                if (!isset($data->$group)) {
                    $data->$group = $group;
                }
            }
        }

        return $data;
    }

    private function updateGroups($saved, $local)
    {
        $savedKeys = array_keys((array) $saved);
        $localKeys = array_keys((array) $local);

        $updatingGroups = array_intersect($savedKeys, $localKeys);

        foreach ($updatingGroups as $group) {
            if (isset($saved->$group) && isset($local->$group)) {
                $savedTemplates = $saved->$group->templates ?? [];
                $localTemplates = $local->$group->templates ?? [];

                $templates = [];

                foreach ($localTemplates as $localTemplate) {
                    $savedTemplate = null;

                    foreach ($savedTemplates as $template) {
                        if ($template->type === $localTemplate->type) {
                            $savedTemplate = $template;
                            break;
                        }
                    }

                    if (empty($savedTemplate)) {
                        $templates[] = $localTemplate;
                        continue;
                    }

                    $subject   = $savedTemplate->subject ?? '';
                    $body      = $savedTemplate->body ?? '';
                    $isEnabled = $savedTemplate->is_enabled ?? false;

                    $template             = $localTemplate;
                    $template->subject    = $subject;
                    $template->body       = $body;
                    $template->is_enabled = $isEnabled;

                    $templates[] = $template;
                }

                $saved->$group            = $local->$group;
                $saved->$group->templates = $templates;
            }
        }

        return $saved;
    }

    private function saveToDatabase($value, $key = 'email_templates')
    {
        $data = (object) [
            'key'   => $key,
            'value' => json_encode($value),
        ];

        $settings       = static::getSettings();
        $settingsData   = $settings->get($key);

        $db = Factory::getContainer()->get(DatabaseInterface::class);

        try {
            if (is_null($settingsData)) {
                $db->insertObject('#__easystore_settings', $data);
            } else {
                $db->updateObject('#__easystore_settings', $data, 'key');
            }
        } catch (Throwable $error) {
            throw $error;
        }
    }
}
