<?php

/**
 * @package     Joomla.Administrator
 * @subpackage  com_easystore
 *
 * @copyright   (C) 2023 - 2024 JoomShaper. <https://www.joomshaper.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace JoomShaper\Component\EasyStore\Administrator\Helper;

use Throwable;
use Joomla\CMS\Factory;
use Joomla\Filesystem\File;
use Joomla\Filesystem\Folder;
use Joomla\Registry\Registry;
use Joomla\Database\DatabaseInterface;
use JoomShaper\Component\EasyStore\Site\Helper\EasyStoreHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Get settings data
 *
 * @since 1.0.0
 */
final class SettingsHelper
{
    /**
     * Get settings value by using key and value
     *
     * $settings = SettingsHelper::getSettings();
     *
     * $currency = $settings->get('general.currency', 'USD:$');
     *
     * @return object
     *
     * @since 1.0.0
     */
    public static function getSettings()
    {
        $db    = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true);
        $query->select($db->quoteName(['key', 'value']))->from($db->quoteName('#__easystore_settings'));
        $db->setQuery($query);

        try {
            $settings = $db->loadObjectList('key') ?? [];

            foreach ($settings as &$setting) {
                $setting = json_decode($setting->value) ?? null;
            }

            unset($setting);

            $settings = (object) $settings;

            return new Registry($settings);
        } catch (Throwable $error) {
            return new Registry([]);
        }
    }

    /**
     * Get the store address details.
     *
     * @return array
     *
     * @since 1.0.2
     */
    public static function getAddress()
    {
        $settings = self::getSettings();

        $countryId = $settings->get('general.country', '');
        $stateId   = $settings->get('general.state', '');

        $CountryCityNames = EasyStoreHelper::getCountryStateFromJson($countryId, $stateId);

        return [
            'address_1' => $settings->get('general.addressLineOne', ''),
            'address_2' => $settings->get('general.addressLineTwo', ''),
            'city'      => $settings->get('general.city', ''),
            'state'     => $CountryCityNames->state,
            'zip_code'  => $settings->get('general.postcode', ''),
            'country'   => $CountryCityNames->country,
            'name'      => $settings->get('general.storeName', ''),
            'phone'     => $settings->get('general.storePhone', ''),
        ];
    }

    public static function getPluginSchema()
    {
        $cachePath = JPATH_CACHE . '/easystore';
        $cacheFile = $cachePath . '/payments.json';

        $url     = 'https://www.joomshaper.com/products/easystore/payments.json';
        $content = '';

        if (!file_exists($cachePath)) {
            Folder::create($cachePath, 0755);
        }

        if (file_exists($cacheFile) && (filemtime($cacheFile) > (time() - (24 * 60 * 60)))) {
            $content = file_get_contents($cacheFile);
        } else {
            $content = file_get_contents($url);

            if (!empty($content)) {
                File::write($cacheFile, $content);
            }
        }

        if (empty($content)) {
            return [];
        }

        return json_decode($content);
    }
}
