<?php

/**
 * @package     Joomla.Administrator
 * @subpackage  com_easystore
 *
 * @copyright   (C) 2023 - 2024 JoomShaper. <https://www.joomshaper.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace JoomShaper\Component\EasyStore\Administrator\Traits;

use JConfig;
use Exception;
use Throwable;
use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use Joomla\Filesystem\Path;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Language\Text;
use Joomla\Registry\Registry;
use Joomla\CMS\Filesystem\File;
use Joomla\CMS\Event\AbstractEvent;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Application\CMSApplication;
use JoomShaper\Component\EasyStore\Site\Lib\Email;
use JoomShaper\Component\EasyStore\Site\Traits\Checkout;
use Joomla\Utilities\ArrayHelper as UtilitiesArrayHelper;
use JoomShaper\Component\EasyStore\Site\Helper\ArrayHelper;
use JoomShaper\Component\EasyStore\Administrator\Model\SettingsModel;
use JoomShaper\Component\EasyStore\Administrator\Helper\SettingsHelper;
use JoomShaper\Component\EasyStore\Administrator\Helper\EasyStoreHelper;

trait AppConfig
{
    use Checkout;
    use Extensions;

    protected $hasDuplicateShipping = false;

    public function appConfig()
    {
        $requestMethod = $this->getInputMethod();

        $this->checkNotAllowedMethods(['POST', 'PUT', 'DELETE', 'PATCH'], $requestMethod);

        if ($requestMethod === 'GET') {
            $this->getAppConfig();
        }
    }

    /**
     * API for getting the app-config data.
     *
     * @return void
     */
    protected function getAppConfig()
    {
        $params      = ComponentHelper::getParams('com_easystore');
        $settings    = SettingsHelper::getSettings();

        $currency      = $settings->get('general.currency', EasyStoreHelper::getDefaultCurrency());
        $currencyChunk = explode(':', $currency, 2);

        $shipping     = (array) $settings->get('shipping', []);
        $shipping     = array_values($shipping);

        $shopPages = $this->getShopPages();

        $settingsModel = new SettingsModel();
        $settingsData  = $settingsModel->getSettings();
        $settingsArray = [];

        foreach ($settingsData as $data) {
            $settingsArray[$data->key] = json_decode($data->value, true);
        }

        $settingsArray['has_products'] = $this->hasProduct() > 0;

        if (!empty($settingsArray['payment']['list'])) {
            foreach ($settingsArray['payment']['list'] as &$item) {
                $item['edit_url']   = (Route::_('index.php?option=com_plugins&task=plugin.edit&extension_id=' . $item['id'], false));
                $item['plugin_id']  = $item['id'];
                $item['has_update'] = $this->getPluginUpdateStatus($item['name']);
                $item['link']       = $this->getPluginUpdateLink($item['name']);

                PluginHelper::importPlugin('easystore', $item['name']);
                $event = AbstractEvent::create(
                    'onBeforePayment',
                    [
                        'subject' => new \stdClass(),
                    ]
                );

                $eventResult           = Factory::getApplication()->getDispatcher()->dispatch($event->getName(), $event);
                $item['is_configured'] = $item['name'] === 'manual_payment' ? true : $eventResult->getArgument('result');
                $item['enabled']       = !$item['is_configured'] ? false : ($item['enabled'] ?? false);
            }

            unset($item);
        } else {
            $settingsArray['payment'] = [
                'list' => [],
            ];
        }

        if (empty($settingsArray['general']['storeEmail'])) {
            /** @var CMSApplication */
            $app                                    = Factory::getApplication();
            $config                                 = $app->getConfig();
            $email                                  = $config->get('mailfrom');
            $settingsArray['general']['storeEmail'] = $email;
        }

        $data = [
            'baseUrl'  => Uri::root(),
            'currency' => [
                'symbol' => $currencyChunk[1],
                'name'   => $currencyChunk[0],
            ],
            'acceptedImageTypes' => $this->getAcceptedImageTypes(),
            'disableAnimation'   => false,
            'settings'           => $settingsArray,
            'shipping'           => array_values($shipping),
            'payment_methods'    => $this->getPaymentMethodList(),
            'shop_pages'         => $shopPages,
            'email'              => $this->getEmailConfigs(),
        ];

        $data['settings']['currency'] = $currencyChunk[1];
        $data['settings']['unit']     = $params->get('product_standard_weight', 'kg');

        $this->sendResponse($data);
    }

    protected function getEmailConfigs()
    {
        /** @var CMSApplication */
        $app    = Factory::getApplication();
        $config = $app->getConfig();

        $fromName  = $config->get('fromname', 'no-reply', 'STRING');
        $fromEmail = $config->get('mailfrom', '', 'EMAIL');

        $settings = SettingsHelper::getSettings();
        $groups   = $settings->get('email_templates', null);

        if (!is_null($groups)) {
            foreach ($groups as &$group) {
                $group->name = Text::_($group->name);

                if (!empty($group->templates)) {
                    foreach ($group->templates as &$template) {
                        $template->title    = Text::_($template->title);
                        $template->subtitle = Text::_($template->subtitle);

                        if (!empty($template->variables)) {
                            foreach ($template->variables as &$variable) {
                                $variable->title = Text::_($variable->title);
                            }

                            unset($variable);
                        }
                    }

                    unset($template);
                }
            }

            unset($group);
        }

        return [
            'sender_name'     => $fromName,
            'sender_email'    => $fromEmail,
            'template_groups' => $groups,
        ];
    }

    protected function hasProduct()
    {
        $db    = Factory::getDbo();
        $query = $db->getQuery(true);
        $query->select('COUNT(id)')
            ->from($db->quoteName('#__easystore_products'))
            ->where($db->quoteName('published') . ' = 1');
        $db->setQuery($query);

        return $db->loadResult() ?? 0;
    }

    /**
     * Get the accepted image extensions.
     *
     * @return array
     */
    protected function getAcceptedImageTypes()
    {
        $mediaParams     = ComponentHelper::getParams('com_media');
        $imageExtensions = $mediaParams->get('image_extensions', '');
        $videoExtensions = $mediaParams->get('video_extensions', '');

        $imageExtensions = array_map(function ($extension) {
            return '.' . $extension;
        }, explode(',', $imageExtensions));

        $videoExtensions = array_map(function ($extension) {
            return '.' . $extension;
        }, explode(',', $videoExtensions));

        return array_merge($imageExtensions, $videoExtensions);
    }

    protected function getShopPages()
    {
        $componentId = ComponentHelper::getComponent('com_easystore')->id ?? null;

        if (empty($componentId)) {
            return [];
        }

        $db    = Factory::getDbo();
        $query = $db->getQuery(true);
        $query->select($db->quoteName(['link', 'title']))
            ->from($db->quoteName('#__menu'))
            ->where($db->quoteName('component_id') . ' = ' . $db->quote($componentId))
            ->where($db->quoteName('published') . ' = 1')
            ->where($db->quoteName('client_id') . ' = 0');
        $db->setQuery($query);

        try {
            $pages = $db->loadObjectList() ?? [];

            foreach ($pages as &$page) {
                $page->label = $page->title;
                $page->value = $page->link;

                unset($page->title, $page->link);
            }

            unset($page);

            return $pages;
        } catch (Throwable $error) {
            return [];
        }

        return [];
    }

    public function saveEmailTemplate()
    {
        $requestMethod = $this->getInputMethod();
        $this->checkNotAllowedMethods(['GET', 'PUT', 'DELETE', 'PATCH'], $requestMethod);

        $group   = $this->getInput('group', '', 'STRING');
        $type    = $this->getInput('type', '', 'STRING');
        $subject = $this->getInput('subject', '', 'STRING');
        $body    = $this->getInput('body', '', 'RAW');

        $settings  = SettingsHelper::getSettings();
        $templates = $settings->get('email_templates');

        if (!isset($templates->$group)) {
            $this->sendResponse(['message' => Text::_('COM_EASYSTORE_EMAIL_TEMPLATE_INVALID_DATA')], 400);
        }

        $group    = $templates->$group;
        $template = ArrayHelper::find(function ($item) use ($type) {
            return $item->type === $type;
        }, $group->templates);

        if (is_null($template)) {
            $this->sendResponse(['message' => Text::_('COM_EASYSTORE_EMAIL_TEMPLATE_INVALID_DATA')], 400);
        }

        $template->subject = $subject;
        $template->body    = $body;

        try {
            $data = (object) [
                'property' => 'email_templates',
                'data'     => json_encode($templates),
            ];
            $this->saveSettings($data);
        } catch (Exception $error) {
            $this->sendResponse(['message' => $error->getMessage()], 500);
        }

        $this->sendResponse(true);
    }

    protected function saveSettings($data)
    {
        try {
            $model  = new SettingsModel();
            $model->createOrUpdate($data);
        } catch (Throwable $error) {
            throw $error;
        }
    }

    public function sendTestEmail()
    {
        $requestMethod = $this->getInputMethod();
        $this->checkNotAllowedMethods(['GET', 'PUT', 'DELETE', 'PATCH'], $requestMethod);

        $receiver = $this->getInput('receiver', '', 'EMAIL');
        $subject  = $this->getInput('subject', '', 'STRING');
        $body     = $this->getInput('body', '', 'RAW');

        $email = new Email($receiver);
        $email->setSubject($subject)->setBody($body);

        try {
            $email->send();
        } catch (Exception $error) {
            $this->sendResponse(['message' => $error->getMessage()], 500);
        }

        $this->sendResponse(true);
    }

    public function updateEmailTemplateStatus()
    {
        $requestMethod = $this->getInputMethod();
        $this->checkNotAllowedMethods(['GET', 'PUT', 'DELETE', 'POST'], $requestMethod);

        $group = $this->getInput('group', '', 'STRING');
        $type  = $this->getInput('type', '', 'STRING');
        $value = $this->getInput('value', 0, 'CMD');

        $settings  = SettingsHelper::getSettings();
        $templates = $settings->get('email_templates');

        if (!isset($templates->$group)) {
            $this->sendResponse(['message' => Text::_('COM_EASYSTORE_EMAIL_TEMPLATE_INVALID_DATA')], 400);
        }

        $group    = $templates->$group;
        $template = ArrayHelper::find(function ($item) use ($type) {
            return $item->type === $type;
        }, $group->templates);

        if (is_null($template)) {
            $this->sendResponse(['message' => Text::_('COM_EASYSTORE_EMAIL_TEMPLATE_INVALID_DATA')], 400);
        }

        $template->is_enabled = (bool) $value;

        try {
            $data = (object) [
                'property' => 'email_templates',
                'data'     => json_encode($templates),
            ];
            $this->saveSettings($data);
        } catch (Exception $error) {
            $this->sendResponse(['message' => $error->getMessage()], 500);
        }

        $this->sendResponse(true);
    }

    public function updateConfiguration()
    {
        $requestMethod = $this->getInputMethod();
        $this->checkNotAllowedMethods(['GET', 'PUT', 'DELETE', 'POST'], $requestMethod);

        $key    = $this->getInput('key', '', 'STRING');
        $value  = $this->getInput('value', '', $key === 'name' ? 'STRING' : 'EMAIL');
        $config = UtilitiesArrayHelper::fromObject(new JConfig());

        $configMap = [
            'name'  => 'fromname',
            'email' => 'mailfrom',
        ];

        if (empty($key) || empty($value)) {
            $this->sendResponse(['message' => Text::_('COM_EASYSTORE_EMAIL_TEMPLATE_INVALID_DATA')], 400);
        }

        if ($config[$configMap[$key]] === $value) {
            $this->sendResponse(true);
        }

        $config[$configMap[$key]] = $value;

        try {
            $this->writeConfigFile(new Registry($config));
        } catch (Exception $error) {
            $this->sendResponse(['message' => $error->getMessage()], 500);
        }

        $this->sendResponse(true);
    }

    /**
     * Method to write the configuration to a file.
     *
     * @param   Registry  $config  A Registry object containing all global config data.
     *
     * @return  int  The task exit code
     *
     * @since  4.1.0
     * @throws \Exception
     */
    private function writeConfigFile(Registry $config)
    {
        // Set the configuration file path.
        $file = JPATH_CONFIGURATION . '/configuration.php';

        // Attempt to make the file writeable.
        if (file_exists($file) && Path::isOwner($file) && !Path::setPermissions($file)) {
            throw new Exception(Text::_('COM_EASYSTORE_CONFIG_FILE_NOT_WRITABLE'));
        }

        try {
            // Attempt to write the configuration file as a PHP class named JConfig.
            $configuration = $config->toString('PHP', ['class' => 'JConfig', 'closingtag' => false]);
            File::write($file, $configuration);
        } catch (Throwable $error) {
            throw $error;
        }

        // Invalidates the cached configuration file
        if (function_exists('opcache_invalidate')) {
            opcache_invalidate($file);
        }

        // Attempt to make the file un-writeable.
        if (Path::isOwner($file) && !Path::setPermissions($file, '0444')) {
            throw new Exception(Text::_('COM_EASYSTORE_CONFIG_FILE_NOT_WRITABLE'));
        }

        return 0;
    }

    /**
     * Get the plugin update status
     *
     * @param string $searchedElement
     * @return mixed
     *
     * @since 1.0.9
     */
    public function getPluginUpdateStatus($searchedElement)
    {
        $installedPlugins = $this->getInstalledPluginList();

        $filteredPlugins = array_filter($installedPlugins, function ($plugins) use ($searchedElement) {
            return $plugins['element'] === $searchedElement;
        });

        return reset($filteredPlugins)['has_update'] ?? false;
    }

    /**
     * Get the plugin update link
     *
     * @param string $searchedElement
     * @return mixed
     *
     * @since 1.0.9
     */
    public function getPluginUpdateLink($searchedElement)
    {
        $installedPlugins = $this->getInstalledPluginList();

        $filteredPlugins = array_filter($installedPlugins, function ($plugins) use ($searchedElement) {
            return $plugins['element'] === $searchedElement;
        });

        return reset($filteredPlugins)['link'] ?? '';
    }
}
