<?php

/**
 * @package     Joomla.Administrator
 * @subpackage  com_easystore
 *
 * @copyright   (C) 2023 - 2024 JoomShaper. <https://www.joomshaper.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace JoomShaper\Component\EasyStore\Administrator\Traits;

use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Filesystem\File;
use Joomla\CMS\Filesystem\Path;
use Joomla\CMS\Filesystem\Folder;
use Joomla\CMS\Helper\ContentHelper;
use Joomla\CMS\User\User as JoomlaUser;
use Joomla\CMS\Component\ComponentHelper;
use JoomShaper\Component\EasyStore\Administrator\Model\UserModel;
use JoomShaper\Component\EasyStore\Administrator\Helper\EasyStoreHelper;
use JoomShaper\Component\EasyStore\Administrator\Helper\EasyStoreDatabaseOrm;

trait User
{
    public function users()
    {
        $search = $this->getInput('search', '', 'STRING');
        $sortBy = $this->getInput('sortBy', '', 'STRING');
        $limit  = $this->getInput('limit', 10, 'INT');
        $offset = $this->getInput('offset', 0, 'INT');
        $all    = (bool) $this->getInput('all', 0);

        $model = new UserModel();

        $params = (object) [
            'search' => $search,
            'sortBy' => $sortBy,
            'limit'  => $limit,
            'offset' => $offset,
            'all'    => $all,
        ];

        $ordering = null;

        if (!empty($params->sortBy)) {
            $ordering = EasyStoreHelper::sortBy($params->sortBy);

            if (\in_array($ordering->field, ['orders', 'spent'])) {
                unset($params->sortBy);
            }
        }

        try {
            $users = $model->getUsers($params);
        } catch (\Exception $error) {
            $this->sendResponse(['message' => $error->getMessage()], 500);
        }

        if (!empty($users->results)) {
            foreach ($users->results as &$user) {
                $user->orders = $model->calculateNumberOfOrders($user->id);
                $user->spent  = $model->calculateUserTotalExpenditure($user->id);
            }

            unset($user);

            if (!empty($ordering) && \in_array($ordering->field, ['orders', 'spent'])) {
                usort($users->results, function ($first, $second) use ($ordering) {
                    $field = $ordering->field;
                    return $ordering->direction === 'ASC' ? (float) $first->$field - (float) $second->$field
                    : (float) $second->$field - (float) $first->$field;
                });
            }
        }

        $this->sendResponse($users);
    }

    public function createUser()
    {
        $canDo          = ContentHelper::getActions('com_easystore');
        $hasPermission  = $canDo->get('core.create') || $canDo->get('core.edit');

        if (!$hasPermission) {
            $this->sendResponse(['message' => Text::_("COM_EASYSTORE_PERMISSION_ERROR_MSG")], 403);
        }

        $name     = $this->getInput('name', '', 'STRING');
        $email    = $this->getInput('email', '', 'STRING');
        $phone    = $this->getInput('phone', '', 'STRING');
        $userType = $this->getInput('user_type', 'customer', 'STRING');

        $isBillingSameAsShipping = (bool) $this->getInput('is_billing_and_shipping_address_same', 0);

        $shippingAddress = $this->getInput('shipping_address', '', 'RAW');
        $billingAddress  = $this->getInput('billing_address', '', 'RAW');

        // Process User image
        $input = Factory::getApplication()->input;
        $file  = $input->files->get('image');

        $imageObject = (object) [
            'name' => null,
            'src'  => null,
        ];

        if (!empty($file)) {
            $isValid = EasyStoreHelper::isValid($file);

            if ($isValid->status) {
                $imageObject = EasyStoreHelper::uploadFile($file, $this->createUserUploadFolder());
            }
        }

        $response = (object) [
            'status'  => false,
            'message' => '',
            'id'      => null,
        ];

        $payload = (object) [
            'user_id'                              => null,
            'phone'                                => $phone,
            'image'                                => $imageObject->src ?? '',
            'user_type'                            => $userType,
            'shipping_address'                     => $shippingAddress,
            'is_billing_and_shipping_address_same' => (int) $isBillingSameAsShipping,
            'billing_address'                      => $isBillingSameAsShipping ? $shippingAddress : $billingAddress,
            'created'                              => Factory::getDate()->toSql(),
            'created_by'                           => Factory::getApplication()->getIdentity()->id,
        ];

        $userData = [
            'id'        => null,
            'name'      => $name,
            'username'  => $email,
            'email'     => $email,
            'password'  => 'secret',
            'password2' => 'secret',
        ];

        try {
            $user = $this->updateOrCreateUser($userData);
        } catch (\Exception $error) {
            $this->sendResponse(['message' => $error->getMessage()], 500);
        }

        $payload->user_id = $user->id;

        try {
            $result            = EasyStoreDatabaseOrm::updateOrCreate('#__easystore_users', $payload, 'id');
            $response->status  = true;
            $response->message = Text::_('COM_EASYSTORE_APP_USER_STORE_SUCCESSFULLY');
            $response->id      = $result->id;

            $this->sendResponse($response, 201);
        } catch (\Exception $e) {
            $response->status  = false;
            $response->message = $e->getMessage();

            $this->sendResponse($response, 500);
        }
    }

    public function updateUser()
    {
        $canDo          = ContentHelper::getActions('com_easystore');
        $hasPermission  = $canDo->get('core.create') || $canDo->get('core.edit');

        if (!$hasPermission) {
            $this->sendResponse(['message' => Text::_("COM_EASYSTORE_PERMISSION_ERROR_MSG")], 403);
        }

        $id = $this->getInput('id', null, 'INT');

        $db    = Factory::getDbo();
        $query = $db->getQuery(true);

        $query->select('user_id');
        $query->from($db->quoteName('#__easystore_users'));
        $query->where($db->quoteName('id') . " = " . $id);

        $db->setQuery($query);
        $userId = $db->loadResult();

        $name     = $this->getInput('name', '', 'STRING');
        $email    = $this->getInput('email', '', 'STRING');
        $phone    = $this->getInput('phone', '', 'STRING');
        $userType = $this->getInput('user_type', 'customer', 'STRING');

        $isBillingSameAsShipping = (bool) $this->getInput('is_billing_and_shipping_address_same', 0);
        $isImageDeleted          = (bool) $this->getInput('is_image_deleted', 0);

        $shippingAddress = $this->getInput('shipping_address', '', 'raw');
        $billingAddress  = $this->getInput('billing_address', '', 'raw');

        // Process User image
        $input = Factory::getApplication()->input;
        $file  = $input->files->get('image');

        // Get existing image
        $orm = new EasyStoreDatabaseOrm();

        $oldImage = $orm->setColumns(['image'])
            ->hasOne($id, '#__easystore_users', 'id')
            ->loadResult();

        $imageObject = (object) [
            'name' => null,
            'src'  => null,
        ];

        if (empty($file)) {
            if ($isImageDeleted) {
                $this->removeImageByPath($oldImage);
            } else {
                $imageObject->src = $oldImage ?? '';
            }
        } else {
            $isValid = EasyStoreHelper::isValid($file);

            if ($isValid->status) {
                $imageObject = EasyStoreHelper::uploadFile($file, $this->createUserUploadFolder());
            }

            // Remove previous Image file
            if (!empty($imageObject->src) && !empty($oldImage)) {
                $this->removeImageByPath($oldImage);
            }
        }

        $response = (object) [
            'status'  => false,
            'message' => '',
            'id'      => null,
        ];

        $userData = [
            'id'    => $userId,
            'name'  => $name,
            'email' => $email,
        ];

        try {
            $this->updateOrCreateUser($userData, true);
        } catch (\Exception $error) {
            $this->sendResponse(['message' => $error->getMessage()], 500);
        }

        $payload = (object) [
            'id'                                   => $id,
            'phone'                                => $phone,
            'image'                                => $imageObject->src ?? '',
            'user_type'                            => $userType,
            'shipping_address'                     => $shippingAddress,
            'is_billing_and_shipping_address_same' => (int) $isBillingSameAsShipping,
            'billing_address'                      => $isBillingSameAsShipping ? $shippingAddress : $billingAddress,
        ];

        try {
            $result = EasyStoreDatabaseOrm::updateOrCreate('#__easystore_users', $payload, 'id');

            $response->status  = true;
            $response->message = Text::_('COM_SPPAGEBUILDER_STORE_USER_UPDATE_SUCCESSFULLY');
            $response->id      = $result->id;

            $this->sendResponse($response, 201);
        } catch (\Exception $e) {
            $response->status  = false;
            $response->message = $e->getMessage();

            $this->sendResponse($response, 500);
        }
    }

    private function updateOrCreateUser($userData, $isUpdate = false)
    {
        $user = new JoomlaUser($userData['id']);

        try {
            $user->bind($userData);
            $user->save();

            if ($user->id && !$isUpdate) {
                $this->addToUserGroupMap($user->id);
            }
        } catch (\Throwable $error) {
            throw $error;
        }

        return $user;
    }

    private function addToUserGroupMap($userId)
    {
        $data = (object) [
            'user_id'  => $userId,
            'group_id' => 2,
        ];

        $db = Factory::getDbo();

        try {
            $db->insertObject('#__user_usergroup_map', $data);
        } catch (\Throwable $error) {
            throw $error;
        }
    }

    public function userById()
    {
        $input = Factory::getApplication()->input;
        $id    = $input->get('id', null, 'INT');
        $model = new UserModel();

        if (is_null($id)) {
            throw new \Exception('The ID is missing!');
        }

        $result = $model->getUserById($id);

        $this->sendResponse($result);
    }

    /**
     * Create Upload Folder for User image upload.
     *
     * @param string $uniqueId  Unique Id
     * @param int $productId    Product Id
     *
     * @return string
     */
    protected function createUserUploadFolder()
    {
        $mediaParams = ComponentHelper::getParams('com_media');
        $directory   = '/user-image';
        $folder      = $mediaParams->get('file_path', 'images') . '/easystore' . $directory;
        $imagePath   = Path::clean(JPATH_ROOT . '/' . $folder);

        if (!Folder::exists($imagePath)) {
            Folder::create($imagePath, 0755);
        }

        return $folder;
    }

    protected function removeImageByPath(string $src)
    {
        if (\file_exists($src)) {
            File::delete($src);
        }
    }
}
