<?php

/**
 * @package     Joomla.Site
 * @subpackage  com_easystore
 *
 * @copyright   (C) 2023 - 2024 JoomShaper. <https://www.joomshaper.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace JoomShaper\Component\EasyStore\Site\Helper;

use DateTime;
use Exception;
use Throwable;
use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Layout\LayoutHelper;
use Joomla\Database\DatabaseInterface;
use Joomla\CMS\Component\ComponentHelper;
use JoomShaper\Component\EasyStore\Site\Model\CartModel;
use JoomShaper\Component\EasyStore\Site\Model\OrderModel;
use JoomShaper\Component\EasyStore\Site\Model\ProductModel;
use JoomShaper\Component\EasyStore\Site\Traits\ProductOption;
use JoomShaper\Component\EasyStore\Administrator\Helper\SettingsHelper;
use JoomShaper\Component\EasyStore\Administrator\Helper\EasyStoreDatabaseOrm;
use JoomShaper\Component\EasyStore\Administrator\Helper\EasyStoreHelper as EasyStoreAdminHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * EasyStore component helper.
 *
 * @since  1.0.0
 */
class EasyStoreHelper
{
    use ProductOption;

    /**
     * Register and load component assets outside of the component scope
     *
     * @return WebAssetManager
     * @since 1.0.0
     */
    public static function wa()
    {
        /** @var CMSApplication */
        $app      = Factory::getApplication();
        $document = $app->getDocument();
        $wa       = $document->getWebAssetManager();
        $registry = $wa->getRegistry();
        $registry->addExtensionRegistryFile('com_easystore');

        $document->addScriptOptions('easystore.base', rtrim(Uri::root(), '/'));

        return $wa;
    }

    public static function attachRequiredAssets()
    {
        static::wa()
            ->useStyle('com_easystore.site')
            ->useStyle('com_easystore.product.site')
            ->useStyle('com_easystore.products.site')
            ->useStyle('com_easystore.cart.drawer.site')
            ->useScript('com_easystore.products.site')
            ->useScript('com_easystore.alpine.site');
    }

    public static function loadLayout($layoutFile, $displayData = [])
    {
        return LayoutHelper::render($layoutFile, $displayData, JPATH_ROOT . '/components/com_easystore/layouts');
    }

    /**
     * Determine whether the checkbox is in a checked state or not.
     *
     * @param   int              $id            The ID of the item to check.
     * @param   array            $checked       Array of all checked ids.
     * @return string|bool
     * @since 1.0.0
     */
    public static function isChecked($id, $checked)
    {
        if (in_array($id, $checked)) {
            return 'checked';
        }

        return false;
    }

    /**
     * Verify whether the user is eligible to review this product.
     *
     * @param  int $userId   User ID
     * @param  int $id       Product ID
     * @since  1.0.0
     * @return bool
     */
    public static function canReview($userId, $id)
    {
        $db    = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true);

        $orm = new EasyStoreDatabaseOrm();

        $customerId = $orm->get('#__easystore_users', 'user_id', $userId, 'id')->loadResult();

        if (!empty($customerId)) {
            $query->select('COUNT(opm.product_id)')
            ->from($db->quoteName('#__easystore_orders', 'o'))
            ->where($db->quoteName('o.customer_id') . " = " . $db->quote($customerId));

            $query->join('LEFT', $db->quoteName('#__easystore_order_product_map', 'opm'), $db->quoteName('opm.order_id') . ' = ' . $db->quoteName('o.id'))
                ->where($db->quoteName('opm.product_id') . " = " . $db->quote($id));

            $db->setQuery($query);

            try {
                return (bool) $db->loadResult();
            } catch (\Throwable $e) {
                throw new \Exception($e->getMessage());
            }
        }


        return false;
    }

    /**
     * Get all the reviews of a product
     *
     * @param  int    $id    Product ID
     * @return mixed         Message object
     * @since  1.0.0
     */
    public static function getReviews($id)
    {
        $db    = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true);

        $query->select($db->quoteName(['a.rating', 'a.subject', 'a.review', 'a.created']))
            ->from($db->quoteName('#__easystore_reviews', 'a'))
            ->where($db->quoteName('a.product_id') . ' = ' . $id)
            ->where($db->quoteName('a.published') . ' = 1');

        // Join over the users.
        $query->select($db->quoteName('uc.name', 'user_name'))
            ->join('LEFT', $db->quoteName('#__users', 'uc'), $db->quoteName('uc.id') . ' = ' . $db->quoteName('a.created_by'));

        $db->setQuery($query);

        try {
            return $db->loadObjectList();
        } catch (\Throwable $e) {
            throw new \Exception($e->getMessage());
        }
    }

    /**
     * Calculate the average rating of a product
     *
     * @param  int   $id                Product ID
     * @param  int   $totalNumOfRating  Total  number of rating of a product
     * @since  1.0.0
     * @return string
     */
    public static function getAverageRating($id, $totalNumOfRating)
    {
        $db    = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true);

        $query->select($db->quoteName('rating'))
            ->from($db->quoteName('#__easystore_reviews'))
            ->where($db->quoteName('product_id') . ' = ' . $id)
            ->where($db->quoteName('published') . ' = 1');

        $db->setQuery($query);

        try {
            $result = $db->loadObjectList();
        } catch (\Throwable $e) {
            throw new \Exception($e->getMessage());
        }

        $averageRating = $totalNumOfRating ? (float) array_sum(array_column($result, 'rating')) / $totalNumOfRating : 0;

        return number_format($averageRating, 1);
    }

    /**
     * Verify if the user has previously provided a review for this product.
     *
     * @param  int $userId   User ID
     * @param  int $id       Product ID
     * @return mixed
     * @since  1.0.0
     */
    public static function hasGivenReview($userId, $id)
    {
        $db    = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true);
        try {
            $query->select('COUNT(*)')
            ->from($db->quoteName('#__easystore_reviews'))
            ->where($db->quoteName('product_id') . ' = ' . $id)
            ->where($db->quoteName('created_by') . ' = ' . $userId)
            ->whereIn($db->quoteName('published'), [0, 1]);

            $db->setQuery($query);
            return $db->loadResult();
        } catch (\Throwable $e) {
            return false;
        }
    }

    /**
     * Calculate total price of an order
     *
     * @param  object  $item
     * @return float
     * @since  1.0.0
     */
    public static function calculateTotalPrice($item)
    {
        $discountedPrice = ($item->discount_value) ? EasyStoreAdminHelper::calculateDiscountedPrice($item->discount_type, $item->discount_value, $item->sub_total) : $item->sub_total;

        $totalPrice = ($discountedPrice + $item->sale_tax + $item->shipping_value - $item->coupon_amount);

        return floatval($totalPrice);
    }

    /**
     * Get all sub category Ids of a category
     *
     * @param  object  $item
     * @return float
     * @since  1.0.0
     */

    public static function getDescendantCategoryIds($catid)
    {
        if ($catid <= 0) {
            $catid = 1;
        }

        $db    = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true);

        $query->select($db->quoteName('id'))
            ->from($db->quoteName('#__easystore_categories'))
            ->where($db->quoteName('parent_id') . ' = ' . $db->quote($catid));

        $db->setQuery($query);

        $results = $db->loadColumn();

        $allChildren = [$catid]; // Include the original $catid

        foreach ($results as $childCatId) {
            $allChildren[] = $childCatId;

            // Recursive call to get children of this child
            $childChildren = static::getDescendantCategoryIds($childCatId);
            $allChildren   = array_merge($allChildren, $childChildren);
        }

        $allChildren = array_unique($allChildren);

        return $allChildren;
    }

    /**
     * Get category of a product
     *
     * @param  int     $id     Product ID
     * @return mixed           Array on success, false on failure.
     * @since  1.0.0
     */
    public static function getCategories($id)
    {
        // Create a new query object.
        $db    = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true);

        $query->select([$db->quoteName('c.id', 'category_id'), $db->quoteName('c.title', 'category_name'), $db->quoteName('c.alias', 'category_alias')])
            ->from($db->quoteName('#__easystore_categories', 'c'))
            ->where($db->quoteName('c.id') . ' = ' . $db->quote($id));

        $db->setQuery($query);

        try {
            return $db->loadAssocList();
        } catch (\Throwable $e) {
            throw new \Exception($e->getMessage());
        }
    }

    /**
     * Get tree of a category
     *
     * @param  int     $id     Category ID
     * @return mixed           Array on success, false on failure.
     * @since  1.0.0
     */
    public static function getCategoryPath($id)
    {
        $db           = Factory::getContainer()->get(DatabaseInterface::class);
        $aliasPath    = [];
        $detailedPath = [];

        while ($id > 0) {
            $query = $db->getQuery(true)
                ->select($db->quoteName(['id', 'title', 'alias', 'parent_id']))
                ->from($db->quoteName('#__easystore_categories'))
                ->where($db->quoteName('id') . ' = ' . (int) $id);

            $db->setQuery($query);
            $result = $db->loadObject();

            if ($result) {
                // Exclude 'root' alias
                if ($result->alias !== 'root') {
                    $aliasPath[$result->id]    = $result->alias;
                    $detailedPath[$result->id] = [
                        'id'        => $result->id,
                        'title'     => $result->title,
                        'alias'     => $result->alias,
                        'parent_id' => $result->parent_id,
                    ];
                }
                $id = $result->parent_id;
            } else {
                // Exit loop if category not found
                break;
            }
        }

        return [
            'aliasPath'    => $aliasPath,
            'detailedPath' => $detailedPath,
        ];
    }

    /**
     * Get tags of a product
     *
     * @param  int        $id     Product ID
     * @return mixed              Object on success, false on failure.
     * @since  1.0.0
     */
    public static function getTags($id)
    {
        // Create a new query object.
        $db    = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true);

        $query->select([$db->quoteName('t.id', 'tag_id'), $db->quoteName('t.alias', 'tag_alias'), $db->quoteName('t.title', 'tag_name')])
            ->from($db->quoteName('#__easystore_tags', 't'));

        $query->join('LEFT', $db->quoteName('#__easystore_product_tag_map', 'product_tag_map') . ' ON product_tag_map.tag_id = t.id')
            ->where($db->quoteName('product_tag_map.product_id') . ' = ' . $db->quote($id));

        $db->setQuery($query);

        try {
            return $db->loadAssocList();
        } catch (\Throwable $e) {
            throw new \Exception($e->getMessage());
        }
    }

    /**
     * Get icon by name
     *
     * @param  string    $iconName
     * @return string
     * @since  1.0.0
     */
    public static function getIcon(string $iconName)
    {
        $path = JPATH_ROOT . '/components/com_easystore/assets/icons/' . $iconName . '.svg';
        if (file_exists($path)) {
            $svg = file_get_contents($path);
            return '<span class="easystore-svg">' . $svg . '</span>';
        }
    }

    /**
     * Get the product image
     *
     * @param  string    $title
     * @return string
     * @since  1.0.0
     */
    public static function setSiteTitle(string $title)
    {
        $app      = Factory::getApplication();
        $siteName = Factory::getConfig()->get('sitename');

        // add site name before or after
        if ($app->get('sitename_pagetitles', 0) == 1) {
            $metaTitle = $siteName . ' - ' . $title;
        } elseif ($app->get('sitename_pagetitles', 0) == 2) {
            $metaTitle = $title . ' - ' . $siteName;
        } else {
            $metaTitle = $title;
        }

        Factory::getDocument()->setTitle($metaTitle);
    }

    /**
     * Display a message
     *
     * @param  string $messageType   Message Type. Ex: error,info,warning
     * @param  string $message       Message to display.
     * @return string
     * @since  1.0.0
     */
    public static function showMessage($messageType, $message)
    {
        $alertClass = ($messageType === 'error') ? "alert-danger" : "alert-$messageType";

        $output = '<div class="alert ' . $alertClass . '">';
        $output .= '<p>' . $message . '</p>';
        $output .= '</div>';

        return $output;
    }

    /**
     * Generate default variants
     *
     * @param int $productId
     * @return string
     */
    public static function generateDefaultVariants($productId)
    {
        $variants = self::getOptions($productId);
        $result   = '';
        if (!empty($variants)) {
            $count = count((array) $variants);
            $i     = 1;

            foreach ($variants as $variant) {
                $result .= $variant->id . ':';
                foreach ($variant->values as $value) {
                    if ($i == $count) {
                        $result .= $value->id;
                    } else {
                        $result .= $value->id . ',';
                    }
                    break;
                }
                $i++;
            }
        }

        return $result;
    }

    /**
     * Initiate EasyStore for SP Page Builder
     * @param string $view
     * @return mixed
     */

    public static function initEasyStore(string $view)
    {
        $app          = Factory::getApplication();
        $input        = $app->input;
        $option       = $input->get('option', '', 'STRING');
        $optionView   = $input->get('view', '', 'STRING');
        $optionLayout = $input->get('layout', 'default', 'STRING');

        $context = $option . '.' . $optionView . '.' . $optionLayout;

        $wa = static::wa();

        if ($view === 'single') {
            $product = null;

            $wa->useStyle('com_easystore.site')
                ->useStyle('com_easystore.product.site');

            if ($context === 'com_sppagebuilder.page.default') {
                $wa->useScript('com_easystore.product.site')
                    ->useScript('com_easystore.wishlist.site')
                    ->useScript('com_easystore.alpine.site');

                $productModel = new ProductModel();

                $db    = Factory::getContainer()->get(DatabaseInterface::class);
                $query = $db->getQuery(true);
                $query->select('id');
                $query->from('#__easystore_products')
                    ->where($db->quoteName('published') . ' = 1');
                $db->setQuery($query);
                $product_id = $db->loadResult();

                if (empty($product_id)) {
                    throw new Exception('Create a product before previewing. No published products available');
                }

                $product = $productModel->getItem((int) $product_id);
            }

            if ($product) {
                return [1, true, ['easystoreItem' => $product, 'easystoreList' => []]];
            }
        }

        if ($view === 'storefront') {
            $wa->useStyle('com_easystore.site')
                ->useStyle('com_easystore.products.site');

            if ($context !== 'com_sppagebuilder.form.edit') {
                $wa->useScript('com_easystore.products.site');
            }
        }

        return [];
    }

    /**
     * Get the page builder content
     *
     * @return object
     * @since  1.0.0
     */
    public static function getPageBuilderData(string $view = 'single')
    {
        if (ComponentHelper::isEnabled('com_sppagebuilder')) {
            $db    = Factory::getDbo();
            $query = $db->getQuery(true)
                ->select('*')
                ->from('#__sppagebuilder')
                ->where([
                    $db->quoteName('extension') . ' = ' . $db->quote('com_easystore'),
                    $db->quoteName('extension_view') . ' = ' . $db->quote($db->escape($view)),
                    $db->quoteName('published') . ' = 1',
                ]);

            $db->setQuery($query);

            return $db->loadObject() ?? false;
        }

        return false;
    }

    /**
     * Get the customer information by Joomla user ID.
     *
     * @param int $userId
     * @return object|null
     */
    public static function getCustomerByUserId($userId)
    {
        if (empty($userId)) {
            return null;
        }

        $db    = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true);

        $query->select(['u.name', 'u.email', 'eu.*'])
            ->from($db->quoteName('#__users', 'u'))
            ->join('LEFT', $db->quoteName('#__easystore_users', 'eu') . ' ON (' . $db->quoteName('eu.user_id') . ' = ' . $db->quoteName('u.id') . ')')
            ->where($db->quoteName('u.id') . ' = ' . (int) $userId);

        $db->setQuery($query);

        try {
            $profile = $db->loadObject() ?? null;
            if ($profile) {
                $profile->avatar = $profile->image ? '<img src="' . $profile->image . '" alt="" class="easystore-profile-image-circle">' : static::getIcon('user');
            }
            return $profile;
        } catch (Throwable $error) {
            return null;
        }
    }


    /**
     * Get the customer by the customer ID
     *
     * @param int $customerId
     * @return object|null
     */
    public static function getCustomerById($customerId)
    {
        if (empty($customerId)) {
            return null;
        }

        $db    = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true);

        $query->select('*')->from($db->quoteName('#__easystore_users'))
            ->where($db->quoteName('id') . ' = ' . $customerId);

        try {
            $db->setQuery($query);

            return $db->loadObject() ?? null;
        } catch (Throwable $error) {
            return null;
        }
    }


    /**
     * Generate responsive columns based on the number of columns
     *
     * @return object
     * @since  1.0.0
     */
    public static function generateResponsiveColumn(int $cols)
    {
        if ($cols < 1 || $cols > 8) {
            $cols = 8;
        }

        // Initialize variables
        $maxPercentage = 100.0;
        $percentages   = [];

        function formatPercentage($percentage)
        {
            return $percentage == round($percentage) ? (int) $percentage : number_format($percentage, 2);
        }

        // Calculate percentages for different screen sizes
        $percentages["xl"] = formatPercentage($maxPercentage / $cols, 2);
        $percentages["lg"] = formatPercentage($maxPercentage / ceil($cols / 2), 2);
        $percentages["md"] = formatPercentage($maxPercentage / ceil($cols / 4), 2);
        $percentages["sm"] = formatPercentage($maxPercentage / 2, 2);
        $percentages["xs"] = formatPercentage($maxPercentage, 2);

        return $percentages;
    }

    /**
     * Extract the name string and make first name and last name.
     *
     * @param string $name
     * @return array
     */
    public static function extractName(string $name)
    {
        $name  = preg_replace("@\s+@", ' ', $name);
        $names = explode(' ', $name);

        switch (count($names)) {
            case 0:
                return ['', ''];
            case 1:
                return [$names[0], ''];
            default:
                $lastName = array_pop($names);
                return [implode(' ', $names), $lastName];
        }

        return ['', ''];
    }

    /**
     * Function to build nested Categories
     *
     * @param array $categories
     * @param int $parentId
     * @return array
     */
    public static function buildNestedCategories($categories, $parentId = 1)
    {
        $nestedCategories = [];

        foreach ($categories as $category) {
            if ($category->cat_parent_id == $parentId) {
                $category->child    = self::buildNestedCategories($categories, $category->cat_id);
                $nestedCategories[] = $category;
            }
        }

        unset($category);

        return $nestedCategories;
    }

    /**
     * Function to get Country Name & State Name from JSON
     *
     * @param string $countryId
     * @param int $stateId
     * @return object
     */
    public static function getCountryStateFromJson($countryId, $stateId)
    {
        $path        = JPATH_ROOT . '/media/com_easystore/data/countries.json';
        $jsonData    = file_get_contents($path);
        $countries   = !empty($jsonData) && is_string($jsonData) ? json_decode($jsonData) : [];
        $countryName = '';
        $stateName   = '';

        foreach ($countries as $country) {
            if ($country->numeric_code == $countryId) {
                $countryName = $country->name;

                foreach ($country->states as $state) {
                    if ($state->id == $stateId) {
                        $stateName = $state->name;
                        break;
                    }
                }

                break;
            }
        }

        return (object) ['country' => $countryName, 'state' => $stateName];
    }

    /**
     * Function to get Country ISO alpha2/alpha3 name bu id
     *
     * @param string $countryId
     * @return object
     */
    public static function getCountryIsoNames($countryId)
    {
        $path        = JPATH_ROOT . '/media/com_easystore/data/countries.json';
        $jsonData    = file_get_contents($path);
        $countries   = !empty($jsonData) && is_string($jsonData) ? json_decode($jsonData) : [];
        $countryName = new \stdClass();

        foreach ($countries as $country) {
            if ($country->numeric_code == $countryId) {
                $countryName->iso2 = $country->alpha_2;
                $countryName->iso3 = $country->alpha_3;
                break;
            }
        }

        return $countryName;
    }

    /**
     * Function to generate Options from JSON
     *
     * @param string $fieldType         Can be 'country', 'state' or 'phoneCode'
     * @param string $stateCountryId    Only need when fieldType is 'state'
     * @return array
     */
    public static function getOptionsFromJson($fieldType = 'country', $stateCountryId = null)
    {
        $path      = JPATH_ROOT . '/media/com_easystore/data/countries.json';
        $jsonData  = file_get_contents($path);
        $data      = !empty($jsonData) && is_string($jsonData) ? json_decode($jsonData) : [];
        $options[] = (object)['name' => Text::_('JSELECT'), 'value' => ''];

        foreach ($data as $country) {
            if ($fieldType === 'country') {
                $options[] = (object) [
                    'name'  => $country->name,
                    'value' => $country->numeric_code,
                ];
            } elseif ($fieldType === 'state') {
                if ($stateCountryId == $country->numeric_code) {
                    foreach ($country->states as $state) {
                        $options[] = (object) [
                            'name'  => $state->name,
                            'value' => $state->id,
                        ];
                    }
                    break;
                }
            } elseif ($fieldType === 'phoneCode') {
                $phoneCode = str_replace(['+', '-'], '', $country->phone_code);
                $options[] = (object) [
                    'name'  => $country->emoji . ' ' . $country->name . ' (' . $phoneCode . ')',
                    'value' => $phoneCode,
                ];
            }
        }

        return $options;
    }

    /**
     * Function to check coupon code validity
     *
     * @param string $startDate
     * @param string $endDate
     * @return bool
     */
    public static function isCouponCodeValid($startDate, $endDate = null)
    {
        $today     = new DateTime('today');
        $startDate = new DateTime($startDate);

        $startDate->setTime(0, 0, 0);
        $today->setTime(0, 0, 0);

        if ($startDate > $today) {
            return false;
        }

        if ($endDate !== null) {
            $endDate = new DateTime($endDate);
            $endDate->setTime(0, 0, 0);

            return $today >= $startDate && $today <= $endDate;
        }

        return $today >= $startDate;
    }


    /**
     * Get Min and Max price
     *
     * @return array
     */
    public static function getPrice()
    {
        // Get the database connection
        $db = Factory::getContainer()->get(DatabaseInterface::class);
        // Create a new query builder object
        $query = $db->getQuery(true);

        // Define the SELECT clause with conditional expressions
        $query->select([
            'IF(p.has_variants = 0, MIN(p.regular_price), MIN(ps.price)) AS min_price',
            'IF(p.has_variants = 0, MAX(p.regular_price), MAX(ps.price)) AS max_price',
        ]);

        // Define the FROM clause and LEFT JOIN
        $query->from($db->quoteName('#__easystore_products', 'p'))
            ->leftJoin($db->quoteName('#__easystore_product_skus', 'ps') . ' ON ' . $db->quoteName('ps.product_id') . ' = ' . $db->quoteName('p.id'));

        // Define the WHERE clause
        $query->where($db->quoteName('p.published') . ' = 1');

        // Define the GROUP BY clause
        $query->group($db->quoteName('p.has_variants'));

        // Execute the query
        $db->setQuery($query);

        // Get the results as an associative array
        $results = $db->loadAssocList();

        // Initialize variables for min and max prices
        $minPrice = null;
        $maxPrice = null;

        // Loop through the results to find the overall min and max prices
        foreach ($results as $result) {
            if ($minPrice === null || $result['min_price'] < $minPrice) {
                $minPrice = (int) $result['min_price'];
            }
            if ($maxPrice === null || $result['max_price'] > $maxPrice) {
                $maxPrice = (int) $result['max_price'];
            }
        }

        return [$minPrice, $maxPrice];
    }

    /**
     * Get cart data in a format which is suitable for payment processing.
     *
     * @param  object $data            - Data information for payment.
     *
     * @return object                  - The formatted cart data object.
     * @since  1.0.0
     */
    public static function getCartDataForPayment($data)
    {
        $backToCheckoutPage = Route::_(Uri::base() . 'index.php?option=com_easystore&view=checkout&cart_token=' . $data->token, false);

        $shippingAddress = is_string($data->shipping_address) ? json_decode($data->shipping_address) : $data->shipping_address;
        $billingAddress  = is_string($data->billing_address) ? json_decode($data->billing_address) : $data->billing_address;

        $cartData        = (new CartModel())->getItem(null, $shippingAddress->country, $shippingAddress->state) ;

        $cartItems       = new \stdClass();
        $settings        = SettingsHelper::getSettings();
        $currencyInfo    = $settings->get('general.currency', EasyStoreAdminHelper::getDefaultCurrency());
        $countryId       = $settings->get('general.country');
        $isTaxEnabled    = $settings->get('checkout.enable_tax', true);

        $chunks              = explode(':', $currencyInfo);
        $currency            = $chunks[0] ?? EasyStoreAdminHelper::getDefaultCurrency('code');
        $currencyNumericCode = self::getCurrencyNumericCode($currency);

        foreach ($cartData->items as $itemData) {
            $items = new \stdClass();

            $items->id               = $itemData->id;
            $items->title            = $itemData->title;
            $items->regular_price    = $itemData->price;
            $items->quantity         = $itemData->quantity;
            $items->discounted_price = $itemData->discounted_price ?: null ;
            $items->image            = $itemData->image->src;

            $cartItems->items[] = $items;
        }

        $cartItems->total_price            = $cartData->total;
        $cartItems->order_id               = $data->order_id;
        $cartItems->store_name             = $settings->get('general.storeName', 'EasyStore');
        $cartItems->tax                    = $isTaxEnabled ? ($cartData->taxable_amount ?: null) : null;
        $cartItems->currency               = $currency;
        $cartItems->country                = static::getCountryIsoNames($countryId);
        $cartItems->shipping_charge        = $cartData->shipping_price ?: null;
        $cartItems->coupon_discount_amount = $cartData->coupon_discount ?: null;
        $cartItems->back_to_checkout_page  = $backToCheckoutPage;
        $cartItems->decimal_separator      = $settings->get('general.decimalSeparator', '.');
        $cartItems->thousands_separator    = $settings->get('general.thousandSeparator', ',');
        $cartItems->currency_numeric_code  = $currencyNumericCode;
        $cartItems->subtotal               = $cartData->sub_total;
        $cartItems->shipping_address       = $shippingAddress;
        $cartItems->billing_address        = $billingAddress;

        if ($data->guest) {
            $cartItems->user_name   = $shippingAddress->name ?? '';
            $cartItems->user_number = $shippingAddress->phone ?? '';
            $cartItems->user_email  = $data->customer_email ?? '';
        } else {
            $customer               = EasyStoreHelper::getCustomerByUserId($data->user_id) ?? new \stdClass();
            $cartItems->user_name   = $customer->name ?? '';
            $cartItems->user_number = $customer->phone ?? '';
            $cartItems->user_email  = $data->customer_email ?? '';
        }

        return $cartItems;
    }

    /**
     * Get order data in a format which is suitable for payment processing.
     *
     * @param  object $data            - Data information for payment.
     *
     * @return object                  - The formatted order data object.
     * @since  1.0.7
     */
    public static function getOrderDataForPayment($data)
    {
        $backToOrderPage = Route::_(Uri::base() . 'index.php?option=com_easystore&view=order&id=' . $data->order_id, false);

        $shippingAddress  = is_string($data->shipping_address) ? json_decode($data->shipping_address) : $data->shipping_address;
        $billingAddress   = is_string($data->billing_address) ? json_decode($data->billing_address) : $data->billing_address;

        $orderData        = (new OrderModel())->getItemForReorder($data->order_id, $shippingAddress->country, $shippingAddress->state);

        $orderItems         = new \stdClass();
        $settings           = SettingsHelper::getSettings();
        $currencyInfo       = $settings->get('general.currency', EasyStoreAdminHelper::getDefaultCurrency());
        $countryId          = $settings->get('general.country');
        $isTaxEnabled       = $settings->get('checkout.enable_tax', true);

        $chunks              = explode(':', $currencyInfo);
        $currency            = $chunks[0] ?? EasyStoreAdminHelper::getDefaultCurrency('code');
        $currencyNumericCode = self::getCurrencyNumericCode($currency);

        foreach ($orderData->items as $itemData) {
            $items = new \stdClass();

            $items->id               = $itemData->product_id;
            $items->title            = $itemData->title;
            $items->regular_price    = $itemData->price;
            $items->quantity         = $itemData->quantity;
            $items->discounted_price = $itemData->discounted_price ?: null ;
            $items->image            = $itemData->image->src;

            $orderItems->items[] = $items;
        }

        $orderItems->total_price            = $orderData->total;
        $orderItems->order_id               = $data->order_id;
        $orderItems->store_name             = $settings->get('general.storeName', 'EasyStore');
        $orderItems->tax                    = $isTaxEnabled ? ($orderData->taxable_amount ?: null) : null;
        $orderItems->currency               = $currency;
        $orderItems->country                = static::getCountryIsoNames($countryId);
        $orderItems->shipping_charge        = $orderData->shipping_price ?: null;
        $orderItems->coupon_discount_amount = $orderData->coupon_discount ?: null;
        $orderItems->back_to_checkout_page  = $backToOrderPage;
        $orderItems->decimal_separator      = $settings->get('general.decimalSeparator', '.');
        $orderItems->thousands_separator    = $settings->get('general.thousandSeparator', ',');
        $orderItems->currency_numeric_code  = $currencyNumericCode;
        $orderItems->subtotal               = $orderData->sub_total;
        $orderItems->shipping_address       = $shippingAddress;
        $orderItems->billing_address        = $billingAddress;


        if ($data->guest) {
            $orderItems->user_name   = $shippingAddress->name ?? '';
            $orderItems->user_number = $shippingAddress->phone ?? '';
            $orderItems->user_email  = $data->customer_email ?? '';
        } else {
            $customer                = EasyStoreHelper::getCustomerByUserId($data->user_id) ?? new \stdClass();
            $orderItems->user_name   = $customer->name ?? '';
            $orderItems->user_number = $customer->phone ?? '';
            $orderItems->user_email  = $data->customer_email ?? '';
        }

        return $orderItems;
    }

    /**
     * Get stock counts of filter
     *
     * @return array
     */
    public static function getStockCounts()
    {
        // Get a database connection.
        $db = Factory::getContainer()->get(DatabaseInterface::class);

        // Create a new query object.
        $query = $db->getQuery(true);

        // CASE conditions for in-stock and out-of-stock items
        $inStockConditions = [
            'a.has_variants = 0 AND a.is_tracking_inventory = 0 AND a.inventory_status = 1',
            'a.has_variants = 0 AND a.is_tracking_inventory = 1 AND a.quantity > 0',
            'a.has_variants = 1 AND a.is_tracking_inventory = 0 AND ps.inventory_status = 1',
            'a.has_variants = 1 AND a.is_tracking_inventory = 1 AND ps.inventory_amount > 0',
        ];

        $outStockConditions = [
            'a.has_variants = 0 AND a.is_tracking_inventory = 0 AND a.inventory_status = 0',
            'a.has_variants = 0 AND a.is_tracking_inventory = 1 AND a.quantity <= 0',
            'NOT EXISTS (SELECT 1 FROM #__easystore_product_skus AS eps WHERE eps.product_id = a.id AND eps.inventory_status = 1 AND a.is_tracking_inventory = 0) AND ((a.has_variants = 1 AND a.is_tracking_inventory = 0 AND ps.inventory_status = 0) OR (a.has_variants = 1 AND a.is_tracking_inventory = 1 AND ps.inventory_amount <= 0))',
        ];

        $query->select([
            "COUNT(DISTINCT CASE WHEN (" . implode(') OR (', $inStockConditions) . ") THEN a.id ELSE NULL END) AS total_in_stock_count",
            "COUNT(DISTINCT CASE WHEN (" . implode(') OR (', $outStockConditions) . ") THEN a.id ELSE NULL END) AS total_out_of_stock_count",
        ])
        ->from($db->quoteName('#__easystore_products', 'a'))
        ->join('LEFT', $db->quoteName('#__easystore_product_skus', 'ps') . ' ON ' . $db->quoteName('ps.product_id') . ' = ' . $db->quoteName('a.id'))
        ->where($db->quoteName('a.published') . ' = 1');

        // Set the query for execution
        $db->setQuery($query);

        // Execute the query and fetch the result
        $results = $db->loadRow();

        // Query only to remove count products with variants with has some variants with 0 quantity
        $query = $db->getQuery(true);
        $query->select("*")
            ->from($db->quoteName('#__easystore_products', 'a'))
            ->where($db->quoteName('a.published') . ' = 1')
            ->where($db->quoteName('a.has_variants') . ' = 1')
            ->where($db->quoteName('a.is_tracking_inventory') . ' = 1');

        $db->setQuery($query);

        $variantProducts = $db->loadObjectList();

        $toRemoveOutOfStockCounter = 0;

        foreach ($variantProducts as $product) {
            $orm      = new EasyStoreDatabaseOrm();
            $variants = $orm->hasMany($product->id, '#__easystore_product_skus', 'product_id')->loadObjectList();

            $count   = count($variants);
            $counter = 0;
            foreach ($variants as $variant) {
                if ($variant->inventory_amount <= 0) {
                    $counter++;
                }
            }

            if ($count != $counter) {
                $toRemoveOutOfStockCounter++;
            }
        }

        $results[1] = $results[1] - $toRemoveOutOfStockCounter;
        $results[1] = $results[1] < 0 ? 0 : $results[1];

        // Return the results
        return $results;
    }

    /**
     * Function for payment status badge color
     *
     * @param string $status
     * @return string
     */
    public static function getPaymentBadgeColor($status)
    {
        if ($status === 'paid') {
            $badgeStatus = "success";
        } elseif ($status === 'unpaid') {
            $badgeStatus = "warning";
        } elseif ($status === 'pending') {
            $badgeStatus = "dark";
        } elseif ($status === 'refunded') {
            $badgeStatus = "info";
        } elseif ($status === 'failed') {
            $badgeStatus = "danger";
        } else {
            $badgeStatus = "secondary";
        }

        return $badgeStatus;
    }

    /**
     * Function to return payment status language string
     *
     * @param string $status
     * @return string
     */
    public static function getPaymentStatusString($status)
    {
        if ($status === 'paid') {
            $result = Text::_('COM_EASYSTORE_PAYMENT_STATUS_PAID');
        } elseif ($status === 'unpaid') {
            $result = Text::_('COM_EASYSTORE_PAYMENT_STATUS_UNPAID');
        } elseif ($status === 'pending') {
            $result = Text::_('COM_EASYSTORE_PAYMENT_STATUS_PENDING');
        } elseif ($status === 'canceled') {
            $result = Text::_('COM_EASYSTORE_PAYMENT_STATUS_CANCELED');
        } elseif ($status === 'failed') {
            $result = Text::_('COM_EASYSTORE_PAYMENT_STATUS_FAILED');
        } elseif ($status === 'refunded') {
            $result = Text::_('COM_EASYSTORE_PAYMENT_STATUS_REFUNDED');
        } else {
            $result = ucfirst(str_replace('_', ' ', $status));
        }

        return $result;
    }

    /**
     * Function to return payment method name
     * @param mixed $method
     * @return string
     */
    public static function getPaymentMethodString($method)
    {
        if ($method === 'manual_payment') {
            $result = Text::_('COM_EASYSTORE_PAYMENT_METHOD_COD');
        } else {
            $result = ucfirst(str_replace('_', ' ', $method));
        }

        return $result;
    }

    /**
     * Get the numeric code of a currency based on its currency code.
     *
     * @param  string   $currency -- The currency code.
     * @return int|null           -- The numeric code of the currency or null if not found.
     * @since  1.0.0
     */

    public static function getCurrencyNumericCode($currency)
    {
        $path     = JPATH_ROOT . '/media/com_easystore/data/currencies.json';
        $jsonData = file_get_contents($path);

        $currencies = !empty($jsonData) && is_string($jsonData) ? json_decode($jsonData) : [];

        if (!empty($currencies)) {
            $index = array_search($currency, array_column($currencies, 'code'));

            if ($index !== false) {
                return $currencies[$index]->numeric_code;
            }
        }

        // Return a default value or handle the case when the currency is not found
        return null;
    }

    /**
     * Function to get order summary data by order id
     * @param int $orderId
     * @return array
     *
     * @since  1.0.9
     */
    public static function getOrderSummary(int $orderId)
    {
        $loggedUser = Factory::getApplication()->getIdentity();
        $user       = EasyStoreHelper::getCustomerByUserId($loggedUser->id);
        $pk         = $orderId;

        $isGuestOrder = 0;
        $guest        = new \stdClass();

        // Create a new query object.
        $db    = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true);

        $query->select(
            [
                $db->quoteName('o.id'),
                $db->quoteName('o.discount_type'),
                $db->quoteName('o.discount_value'),
                $db->quoteName('o.shipping'),
                $db->quoteName('o.tracking_number'),
                $db->quoteName('o.transaction_id'),
                $db->quoteName('o.shipping_carrier'),
                $db->quoteName('o.tracking_url'),
                $db->quoteName('o.coupon_type'),
                $db->quoteName('o.coupon_code'),
                $db->quoteName('o.coupon_amount'),
                $db->quoteName('o.sale_tax'),
                $db->quoteName('o.shipping_type'),
                $db->quoteName('o.shipping_type'),
                $db->quoteName('o.shipping_value'),
                $db->quoteName('o.payment_method'),
                $db->quoteName('o.is_guest_order'),
                $db->quoteName('o.customer_id'),
                $db->quoteName('o.customer_email'),
                $db->quoteName('o.shipping_address'),
                $db->quoteName('o.billing_address'),
            ]
        )
            ->from($db->quoteName('#__easystore_orders', 'o'))
            ->where($db->quoteName('o.id') . ' = ' . $pk)
            ->where($db->quoteName('o.published') . ' = 1');

        $db->setQuery($query);

        try {
            $item = $db->loadObject();

            $isGuestOrder = $item->is_guest_order;

            if (empty($isGuestOrder) && is_null($item->customer_id) || empty($isGuestOrder) && is_null($user)) {
                throw new Exception(Text::_("COM_EASYSTORE_ORDER_NOT_FOUND"), 404);
            }

            if (!empty($isGuestOrder)) {
                $guest->name                                 = '';
                $guest->email                                = $item->customer_email;
                $guest->is_billing_and_shipping_address_same = false;
            }

            $query->clear();

            $subquery = $db->getQuery(true);
            $subquery->select($db->quoteName(['order_id', 'discount_type', 'discount_value', 'price', 'quantity']))
                ->select('SUM(' . $db->quoteName('price') . ' * ' . $db->quoteName('quantity') . ') AS sub_total')
                ->from($db->quoteName('#__easystore_order_product_map'))
                ->where($db->quoteName('order_id') . ' = ' . $pk)
                ->group([$db->quoteName('order_id'), $db->quoteName('discount_type'), $db->quoteName('discount_value'), $db->quoteName('price'), $db->quoteName('quantity')]);

            // Build the main query
            $query->select($db->quoteName(['order_id', 'discount_type', 'discount_value', 'price', 'quantity', 'sub_total']))
                ->select('
                CASE WHEN ' . $db->quoteName('discount_value') . ' > 0 THEN
                    CASE 
                        WHEN ' . $db->quoteName('discount_type') . ' = ' . $db->quote('percent') . ' THEN 
                        ' . $db->quoteName('price') . ' - (' . $db->quoteName('price') . ' * (' . $db->quoteName('discount_value') . ') / 100)
                        ELSE ' . $db->quoteName('price') . ' - ' . $db->quoteName('discount_value') . '
                    END 
                ELSE
                    0.00
                END AS discounted_price,

                CASE
                    WHEN (
                        CASE
                            WHEN ' . $db->quoteName('discount_value') . ' > 0 THEN
                                CASE
                                    WHEN ' . $db->quoteName('discount_type') . ' = "percent" THEN
                                       ' . $db->quoteName('price') . ' - (' . $db->quoteName('price') . ' * (' . $db->quoteName('discount_value') . ') / 100)
                                    ELSE ' . $db->quoteName('price') . ' - ' . $db->quoteName('discount_value') . '
                                END
                            ELSE
                                0.00
                        END
                    ) = 0.00 THEN 0.00
                    ELSE
                    ' . $db->quoteName('price') . ' - (
                            CASE
                                WHEN ' . $db->quoteName('discount_value') . ' > 0 THEN
                                    CASE
                                        WHEN ' . $db->quoteName('discount_type') . ' = "percent" THEN
                                           ' . $db->quoteName('price') . ' - (' . $db->quoteName('price') . ' * (' . $db->quoteName('discount_value') . ') / 100)
                                        ELSE
                                           ' . $db->quoteName('price') . ' - ' . $db->quoteName('discount_value') . '
                                    END
                                ELSE
                                    0.00
                            END
                        )
                END  AS special_discounted_amount
                ')
                ->from('(' . $subquery . ') AS subquery');

            // Execute the query
            $db->setQuery($query);

            $orderedProducts = $db->loadObjectList();


            $item->sub_total                 = 0.00;
            $item->discounted_sub_total      = 0.00;
            $item->special_discounted_amount = 0.00;
            $item->extra_discount_amount     = 0.00;
            $item->shipping_cost             = 0.00;
            $item->sub_total_tax             = 0.00;
            $item->coupon_discount           = 0.00;
            $total_price                     = 0.00;


            if (!empty($orderedProducts)) {
                foreach ($orderedProducts as &$product) {
                    $item->sub_total += floatval($product->sub_total);

                    if (floatval($product->discounted_price) > 0) {
                        $item->discounted_sub_total += floatval($product->discounted_price);
                    }

                    if (!is_null(floatval($product->special_discounted_amount))) {
                        $item->special_discounted_amount += floatval($product->special_discounted_amount);
                    }

                    $item->sub_total -= $item->special_discounted_amount;
                }

                unset($product);
            }

            if (!is_null($item->coupon_code) && floatval($item->coupon_amount) > 0) {
                $item->coupon_discount = EasyStoreAdminHelper::calculateDiscountValue($item->coupon_type, $item->coupon_amount, $item->sub_total);
            }

            if (!is_null($item->discount_value) && floatval($item->discount_value) > 0) {
                $extra_discount = EasyStoreAdminHelper::calculateDiscountedPrice($item->discount_type, $item->discount_value, $item->sub_total);

                $item->extra_discount_amount = $item->sub_total - $extra_discount;
            }

            if (!empty($item->shipping) && is_string($item->shipping)) {
                // Convert the shipping string to an object
                $shippingData = json_decode($item->shipping);

                if (isset($shippingData->offerFreeShipping) && $shippingData->offerFreeShipping) {
                    // Check if there's an offer on a specific amount
                    $offerOnAmount = (float) ($shippingData->offerOnAmount ?? null);

                    if ($item->sub_total > $offerOnAmount) {
                        // Apply free shipping if the subtotal is above the offer amount
                        $shippingData->rate = 0;
                    }
                }

                if (isset($shippingData->rate)) {
                    // Format the shipping rate with currency
                    $shippingData->rate_with_currency = EasyStoreAdminHelper::formatCurrency($shippingData->rate);
                }

                if (isset($shippingData->rate) && empty($shippingData->rate)) {
                    // Set the shipping name to "Free Shipping" if the rate is empty
                    $shippingData->name = Text::_('COM_EASYSTORE_FREE_SHIPPING');
                }

                // Update the shipping cost with the calculated rate or default to 0
                $item->shipping_cost = $shippingData->rate ?? 0;
            }


            if (!empty($item->sale_tax)) {
                $item->sub_total_tax = $item->sub_total + floatval($item->sale_tax);
            }

            $total_price = $item->sub_total + $item->shipping_cost + $item->sale_tax - $item->coupon_discount - $item->extra_discount_amount;

            $orderModel                                    = new OrderModel();
            $item->products                                = $orderModel->getProducts($item->id);
            $item->sub_total_with_currency                 = EasyStoreAdminHelper::formatCurrency($item->sub_total);
            $item->sub_total_tax_with_currency             = EasyStoreAdminHelper::formatCurrency($item->sub_total_tax);
            $item->special_discounted_amount_with_currency = EasyStoreAdminHelper::formatCurrency($item->special_discounted_amount);
            $item->extra_discount_amount_with_currency     = EasyStoreAdminHelper::formatCurrency($item->extra_discount_amount);
            $item->total_price_with_currency               = EasyStoreAdminHelper::formatCurrency($total_price);
            $item->shipping_cost_with_currency             = EasyStoreAdminHelper::formatCurrency($item->shipping_cost);
            $item->coupon_discount_with_currency           = EasyStoreAdminHelper::formatCurrency($item->coupon_discount);
            $item->sale_tax_with_currency                  = EasyStoreAdminHelper::formatCurrency($item->sale_tax);

            foreach ($item->products as &$product) {
                $product->price_with_currency          = EasyStoreAdminHelper::formatCurrency($product->price);
                $product->discount_price_with_currency = EasyStoreAdminHelper::formatCurrency(EasyStoreAdminHelper::calculateDiscountedPrice($product->discount_type, $product->discount_value, $product->price));
                $product->total_price                  = floatval($product->quantity * $product->price);
                $product->total_price_with_currency    = EasyStoreAdminHelper::formatCurrency($product->total_price);


                unset($product->media);
            }

            unset($product);

            if (empty($isGuestOrder) && !empty($user->shipping_address)) {
                $user->shipping_address = $orderModel->generateAddress($user->shipping_address);
            } else {
                $guest->shipping_address = $orderModel->generateAddress($item->shipping_address);
            }

            if (empty($isGuestOrder) && !empty($user->billing_address)) {
                $user->billing_address = $orderModel->generateAddress($user->billing_address);
            } else {
                $guest->billing_address = $orderModel->generateAddress($item->billing_address);
            }

            $item->customerData = empty($isGuestOrder) ? $user : $guest;

            return (array) $item;
        } catch (\Throwable $e) {
            throw new \Exception($e->getMessage());
        }
    }

    /**
     * Converts the country and state code values in the given JSON-encoded address to their corresponding string names.
     * @param string $addressJson   Should be JSON data
     * @return void
     * @since  1.0.9
     */
    public static function convertAddressCountryStateData(string $addressJson)
    {
        $address            = json_decode($addressJson, true);
        $CountryStateNames  = EasyStoreHelper::getCountryStateFromJson($address['country'], $address['state']);
        $address['country'] = $CountryStateNames->country;
        $address['state']   = $CountryStateNames->state;

        return $address;
    }
}
