<?php

/**
 * @package     EasyStore.Site
 * @subpackage  com_easystore
 *
 * @copyright   Copyright (C) 2023 - 2024 JoomShaper <https://www.joomshaper.com>. All rights reserved.
 * @license     GNU General Public License version 3; see LICENSE
 */

namespace JoomShaper\Component\EasyStore\Site\Model;

use Exception;
use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Filesystem\Path;
use Joomla\CMS\MVC\Model\ItemModel;
use Joomla\Database\DatabaseInterface;
use JoomShaper\Component\EasyStore\Site\Helper\ArrayHelper;
use JoomShaper\Component\EasyStore\Site\Controller\CartController;
use JoomShaper\Component\EasyStore\Administrator\Helper\SettingsHelper;
use JoomShaper\Component\EasyStore\Administrator\Helper\EasyStoreDatabaseOrm;
use JoomShaper\Component\EasyStore\Administrator\Helper\EasyStoreHelper as AdministratorEasyStoreHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects


class CartModel extends ItemModel
{
    /**
     * Model context string.
     *
     * @var        string
     */
    protected $_context = 'com_easystore.cart';

    public function getItem($shippingId = null, $country = null, $state = null)
    {
        $orm                 = new EasyStoreDatabaseOrm();
        $settings            = SettingsHelper::getSettings();
        $weightUnit          = $settings->get('products.standardUnits.weight', 'kg');
        $isTaxEnabled        = $settings->get('checkout.enable_tax', true);
        $isCouponCodeEnabled = $settings->get('checkout.enable_coupon_code', true);

        $db    = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true);
        $token = $this->getToken();

        $conditions = [
            $db->quoteName('token') . ' = ' . $db->quote($token),
        ];

        $query->select('*')->from($db->quoteName('#__easystore_cart'))
            ->where($conditions);
        $db->setQuery($query);

        try {
            $cart = $db->loadObject();
        } catch (Exception $error) {
            $cart = null;
        }

        if (empty($cart)) {
            return null;
        }

        $columns = [
            'ci.quantity',
            'product.id as product_id',
            'product.title',
            'product.alias',
            'product.regular_price',
            'product.has_sale',
            'product.quantity as available_quantity',
            'product.is_tracking_inventory',
            'product.inventory_status as product_inventory_status',
            'product.quantity as product_inventory_amount',
            'product.discount_type',
            'product.discount_value',
            'product.enable_out_of_stock_sell',
            'product.has_variants',
            'product.weight',
            'sku.id as sku_id',
            'sku.combination_value',
            'sku.price as price',
            'sku.weight as sku_weight',
            'sku.image_id',
            'sku.inventory_status as sku_inventory_status',
            'sku.inventory_amount as sku_inventory_amount',
        ];

        $cart->items = $orm->setColumns(['id'])
            ->hasMany($cart->id, ['#__easystore_cart_items', 'ci'], 'cart_id')
            ->updateQuery(function ($query) use ($db, $columns) {
                $query->select($columns)->join(
                    'LEFT',
                    $db->quoteName('#__easystore_products', 'product') . ' ON ' . $db->quoteName('product.id') . ' = ' . $db->quoteName('ci.product_id')
                )->join(
                    'LEFT',
                    $db->quoteName('#__easystore_product_skus', 'sku') . ' ON ' . $db->quoteName('sku.id') . ' = ' . $db->quoteName('ci.sku_id')
                );
            })->loadObjectList();

        $cart->sub_total    = 0;
        $cart->total_weight = 0;

        if (!empty($cart->items)) {
            foreach ($cart->items as &$item) {
                if (!empty($item->image_id)) {
                    $item->image = $orm->setColumns(['id', 'src'])
                        ->hasOne($item->image_id, '#__easystore_media', 'id')
                        ->loadObject();
                } else {
                    $item->image = $orm->setColumns(['id', 'src'])
                        ->hasOne($item->product_id, '#__easystore_media', 'product_id')
                        ->updateQuery(function ($query) use ($db) {
                            $query->where($db->quoteName('is_featured') . ' = 1');
                        })->loadObject();
                }

                if (!empty($item->image) && !empty($item->image->src)) {
                    $item->image->src = Path::clean(Uri::root(true) . '/' . $item->image->src);
                }

                if ($item->has_variants) {
                    $item->inventory_amount = $item->sku_inventory_amount ?? 0;
                    $item->inventory_status = $item->sku_inventory_status ?? 0;
                    $item->price            = $item->price ?? $item->regular_price;

                    $weight                 = $item->sku_weight ?? 0;
                    $item->weight           = (float) $weight;
                    $item->weight_with_unit = $item->sku_weight . $weightUnit;
                } else {
                    $item->price            = $item->regular_price ?? 0;
                    $item->inventory_amount = $item->available_quantity ?? 0;
                    $item->inventory_status = $item->product_inventory_status ?? 0;

                    $weight                 = $item->weight ?? 0;
                    $item->weight           = (float) $weight;
                    $item->weight_with_unit = $item->weight . $weightUnit;
                }

                $cart->total_weight += $item->weight * (int) $item->quantity;

                $item->discount = (object) [
                    'type'   => $item->discount_type ?? 'percent',
                    'amount' => $item->discount_value ?? 0,
                ];

                unset(
                    $item->discount_type,
                    $item->discount_value,
                    $item->regular_price,
                    $item->sku_inventory_amount,
                    $item->product_inventory_amount,
                    $item->sku_inventory_status,
                    $item->product_inventory_status,
                    $item->available_quantity,
                    $item->image_id
                );

                $item->discounted_price = 0;
                $item->item_price       = 0;
                $item->item_price       = $item->price;

                if ($item->has_sale && $item->discount->amount > 0) {
                    $item->discounted_price = AdministratorEasyStoreHelper::calculateDiscountedPrice($item->discount->type, $item->discount->amount, $item->price);
                    $item->item_price       = $item->discounted_price;
                }

                $item->total_price                    = floatval($item->discounted_price) > 0 ? $item->discounted_price * $item->quantity : $item->price * $item->quantity;
                $item->total_price_with_currency      = AdministratorEasyStoreHelper::formatCurrency($item->total_price);
                $item->price_with_currency            = AdministratorEasyStoreHelper::formatCurrency($item->price);
                $item->discounted_price_with_currency = AdministratorEasyStoreHelper::formatCurrency($item->discounted_price);

                $item->options = $this->detectProductOptionFromCombination(
                    $this->getProductOptionsById($item->product_id),
                    $item->combination_value
                );

                $cart->sub_total += $item->total_price ?? 0;
            }

            unset($item);
        }

        $settingsModel   = new SettingsModel();
        $taxRate         = $settingsModel->getTaxRate($country, $state);
        $shippingMethods = $settingsModel->getShipping($country, $state);

        if (!empty($cart->shipping_method) && is_string($cart->shipping_method)) {
            $cart->shipping_method = json_decode($cart->shipping_method);
        }

        if (!empty($shippingId) && !empty($shippingMethods)) {
            $method = ArrayHelper::find(function ($item) use ($shippingId) {
                return $item->uuid === $shippingId;
            }, $shippingMethods);

            $cart->shipping_method = $method;
        }

        if (empty($cart->shipping_method) && !empty($shippingMethods)) {
            $cart->shipping_method = reset($shippingMethods);
        }

        if (isset($cart->shipping_method->offerFreeShipping) && $cart->shipping_method->offerFreeShipping) {
            $offerOnAmount = $cart->shipping_method->offerOnAmount ?? null;

            if (!is_null($offerOnAmount)) {
                $offerOnAmount = (float) $offerOnAmount;

                if ($cart->sub_total > $offerOnAmount) {
                    $cart->shipping_method->rate = 0;
                }
            }
        }

        if (isset($cart->shipping_method->rate)) {
            $cart->shipping_method->rate_with_currency = AdministratorEasyStoreHelper::formatCurrency($cart->shipping_method->rate);
        }

        if (isset($cart->shipping_method->rate) && empty($cart->shipping_method->rate)) {
            $cart->shipping_method->name = Text::_('COM_EASYSTORE_FREE_SHIPPING');
        }

        $cart->shipping_price  = $cart->shipping_method->rate ?? 0;
        $cart->coupon_discount = 0;
        $cart->coupon_amount   = (float) $cart->coupon_amount;

        if (!empty($cart->coupon_code) && $cart->coupon_amount > 0) {
            $cart->coupon_discount = AdministratorEasyStoreHelper::calculateDiscountValue($cart->coupon_type, $cart->coupon_amount, $cart->sub_total);
        }

        $cart->tax_rate  = !empty($taxRate) ? $taxRate->rate : 0;
        $isTaxOnShipping = !empty($taxRate) ? $taxRate->applyOnShipping : false;

        $cart->coupon_discount_with_currency = AdministratorEasyStoreHelper::formatCurrency($cart->coupon_discount);
        $cart->sub_total_with_currency       = AdministratorEasyStoreHelper::formatCurrency($cart->sub_total);

        if ($isTaxOnShipping) {
            $totalPriceWithShipping = $cart->sub_total + $cart->shipping_price;
            $cart->taxable_amount   = ($totalPriceWithShipping * $cart->tax_rate) / 100;
        } else {
            $cart->taxable_amount = ($cart->sub_total * $cart->tax_rate) / 100;
        }

        $cart->total = $cart->sub_total + $cart->shipping_price;

        if ($isTaxEnabled) {
            $cart->total += $cart->taxable_amount;
        }

        if ($isCouponCodeEnabled) {
            $cart->total -= $cart->coupon_discount;
        }

        $cart->taxable_amount_with_currency  = AdministratorEasyStoreHelper::formatCurrency($cart->taxable_amount);
        $cart->total_with_currency           = AdministratorEasyStoreHelper::formatCurrency($cart->total);
        $cart->total_weight_with_unit        = $cart->total_weight . $weightUnit;

        return $cart;
    }

    public function calculateTotalWeight()
    {
        $db    = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true);
        $token = $this->getToken();

        $conditions = [
            $db->quoteName('token') . ' = ' . $db->quote($token),
        ];

        $query->select('id')->from($db->quoteName('#__easystore_cart'))
            ->where($conditions);
        $db->setQuery($query);

        $cartId = 0;

        try {
            $cartId = $db->loadResult();
        } catch (Exception $error) {
            return 0;
        }

        $orm     = new EasyStoreDatabaseOrm();
        $columns = ['product.weight as product_weight', 'sku.weight as sku_weight', 'ci.quantity'];
        $items   = $orm->setColumns(['product_id', 'sku_id'])
            ->hasMany($cartId, ['#__easystore_cart_items', 'ci'], 'cart_id')
            ->updateQuery(function ($query) use ($db, $columns) {
                $query->select(($columns))->join(
                    'LEFT',
                    $db->quoteName('#__easystore_products', 'product') . ' ON ' . $db->quoteName('product.id') . ' = ' . $db->quoteName('ci.product_id')
                )->join(
                    'LEFT',
                    $db->quoteName('#__easystore_product_skus', 'sku') . ' ON ' . $db->quoteName('sku.id') . ' = ' . $db->quoteName('ci.sku_id')
                );
            })->loadObjectList() ?? [];

        if (empty($items)) {
            return 0;
        }

        $weight = 0;

        foreach ($items as $item) {
            if (!is_null($item->sku_id)) {
                $skuWeight = $item->sku_weight ?? 0;
                $weight += (float) $skuWeight * (int) $item->quantity;
            } else {
                $productWeight = $item->product_weight ?? 0;
                $weight += (float) $productWeight * (int) $item->quantity;
            }
        }

        return $weight;
    }

    public function detectProductOptionFromCombination(array $options, $combination)
    {
        if (empty($options) || empty($combination)) {
            return [];
        }

        $combinationArray = explode(';', $combination);

        $result = [];

        if (!empty($combinationArray)) {
            foreach ($combinationArray as $com) {
                foreach ($options as $option) {
                    if (!empty($option->values)) {
                        foreach ($option->values as $value) {
                            if (strtolower($com) === strtolower($value->name)) {
                                $result[] = (object) [
                                    'key'   => $option->name,
                                    'name'  => $value->name,
                                    'color' => $value->color ?? '',
                                ];
                            }
                        }
                    }
                }
            }
        }

        return $result;
    }

    public function getProductOptionsById(int $productId)
    {
        $db    = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true);

        $query->select('*')->from($db->quoteName('#__easystore_product_options'))
            ->where($db->quoteName('product_id') . ' = ' . $productId);

        $db->setQuery($query);

        try {
            $options = $db->loadObjectList() ?? [];
        } catch (\Exception $error) {
            $options = [];
        }

        $orm = new EasyStoreDatabaseOrm();

        foreach ($options as &$option) {
            $option->values = $orm->hasMany($option->id, '#__easystore_product_option_values', 'option_id')->loadObjectList();
        }

        unset($option);

        return $options;
    }

    protected function getToken()
    {
        $controller = new CartController();

        return $controller->hasToken() ? $controller->getToken() : null;
    }

    public function updateStatus(string $status)
    {
        $orm = new EasyStoreDatabaseOrm();

        $data = (object) [
            'token'  => $this->getToken(),
            'status' => $status,
        ];

        try {
            $orm->update('#__easystore_cart', $data, 'token');
        } catch (\Throwable $error) {
            throw $error;
        }
    }
}
