/**
 * @copyright   Copyright (C) 2020 JoomShaper. All rights reserved.
 * @license     GNU General Public License version 3; see LICENSE
 * @since      1.0.0
 */
function debounce(func, delay) {
    let timeout;

    return function (...args) {
        clearTimeout(timeout);

        timeout = setTimeout(() => {
            func.apply(this, args);
        }, delay);
    };
}

document.addEventListener('alpine:init', () => {
    const cartItemCount = Alpine.reactive({
        total: 0,
    });
    Alpine.data('easystore_cart', () => ({
        async init() {
            await this.fetchCartData();

            cartItemCount.total = this.getTotalNumberOfItems();

            Alpine.effect(() => {
                this.itemCount = cartItemCount.total;
                const footer = document.querySelector('[easystore-drawer-footer]');

                if (!footer) {
                    return;
                }

                if (this.itemCount === 0) {
                    footer.classList.add('easystore-hide');
                } else {
                    footer.classList.remove('easystore-hide');
                }
            });
        },

        isLoading: false,
        loading: false,
        item: { items: [] },
        stockMessage: [],
        itemCount: 0,
        validVideoFormats: ['mp4', 'avi', 'mov', 'mkv', 'flv', 'wmv', 'webm'],

        getTotalNumberOfItems() {
            if (!this.item.items?.length === 0) {
                return 0;
            }

            return this.item.items.reduce((total, current) => {
                return total + Number(current.quantity);
            }, 0);
        },

        getFileExtension(filename = '') { 
            return filename.split('.').pop();
        },

        async fetchCartData() {
            this.loading = true;
            const response = await fetch(
                `${Joomla.getOptions('easystore.base')}/index.php?option=com_easystore&task=cart.getCartData`
            );
            const data = await response.json();
            setTimeout(() => {
                this.loading = false;
            }, 1000);

            if (!!data.data) {
                this.item = {
                    ...this.item,
                    ...data.data,
                };

                this.item.items.forEach(item => {
                    item.isVideo = this.validVideoFormats.includes(this.getFileExtension(item?.image?.src));
                });
            }
        },
        decrementQuantity(index) {
            if (this.isLoading) return;

            this.item.items[index].quantity = Math.max(0, this.item.items[index].quantity - 1);
            this.requestForUpdate(this.item.items);
        },
        incrementQuantity(index) {
            if (this.isLoading) return;

            const item = this.item.items[index];

            if (item.is_tracking_inventory) {
                if (item.inventory_amount <= 0) {
                    return;
                }
                this.item.items[index].quantity = Math.min(
                    item.inventory_amount,
                    Number(this.item.items[index].quantity) + 1
                );
            } else {
                this.item.items[index].quantity = Number(this.item.items[index].quantity) + 1;
            }

            this.requestForUpdate(this.item.items);
        },

        handleQuantityChange(event, index) {
            let value = Number(event.target.value);

            if (isNaN(value)) {
                return;
            }

            value = Math.max(1, value);

            this.item.items[index].quantity = value;
            this.requestForUpdate(this.item.items);
        },
        removeCartItem(index) {
            if (this.item.items.length > 0) {
                this.item.items.splice(index, 1);
            }

            this.requestForUpdate(this.item.items);
        },
        toCurrency(number) {
            return number.toLocaleString();
        },
        requestForUpdate: debounce(function (data) {
            this.isLoading = true;
            const formData = new FormData();
            const items = data
                .map(item => {
                    return {
                        product_id: item.product_id,
                        sku_id: item.sku_id ?? null,
                        quantity: item.quantity ?? 0,
                        id: item.id ?? null,
                    };
                })
                .filter(item => item.quantity > 0);

            formData.append('items', JSON.stringify(items));
            formData.append('initiator', 'cart-page');
            formData.append('_method', 'POST');

            const url = `${Joomla.getOptions('easystore.base')}/index.php?option=com_easystore&task=cart.addToCart`;

            Joomla.request({
                url,
                method: 'POST',
                data: formData,
                onSuccess: async response => {
                    response = JSON.parse(response);

                    if (!!response.data) {
                        await this.fetchCartData();
                        cartItemCount.total = this.getTotalNumberOfItems();
                    }

                    if (this.item.items?.length === 0) {
                        window.location.reload();
                    }
                },
                onComplete: () => {
                    setTimeout(() => {
                        this.isLoading = false;
                    }, 1000);
                },
                onError: error => {
                    const response = error.response && JSON.parse(error.response);
                    errorMessage = response?.data?.message ?? '';
                    alert(errorMessage);
                },
            });
        }, 500),
    }));
});
