/**
 * @copyright   Copyright (C) 2020 JoomShaper. All rights reserved.
 * @license     GNU General Public License version 3; see LICENSE
 * @since      1.0.0
 */

Joomla = window.Joomla || {};
((Joomla, document) => {
    document.addEventListener('DOMContentLoaded', function () {
        const defaultQuantity = 1;
        let subTotalAmount, discountAmount, shippingAmount, taxAmount, couponAmount, totalAmount;
        setPriceInfo();


        document.addEventListener('input', function (event) {
            // Product quantity change functionality
            if (event.target.classList.contains('product-quantity')) {
                const quantityInput = event.target;
                if (!quantityInput.value || quantityInput.value < defaultQuantity) {
                    quantityInput.value = defaultQuantity;
                }

                let quantity = quantityInput.value;
                const totalAmountEl =
                    quantityInput?.parentElement?.nextElementSibling?.querySelector('.quantity-price');
                totalAmountEl.value = Number(totalAmountEl.dataset.value) * Number(quantity);
                setPriceInfo();
            }

            if (event.target.id.includes('jform_discount_type')) {
                setPriceInfo();
            }

            if (event.target.id.includes('jform_discount_value')) {
                setPriceInfo();
            }
        });

        document.addEventListener('change', function (event) {
            if (event.target.parentNode.parentNode.parentNode.classList.contains('easystore-shipping-method')) {
                const targetValue = event.target.value;
                if (targetValue) {
                    const chunks = targetValue.split(':');
                    document.querySelector('.shipping-amount').value = chunks[0];
                }
                setPriceInfo();
            }
        });

        function getSubTotalPrice() {
            let items = document.querySelectorAll('.product-item .quantity-price');
            let subTotalPrice = 0.0;

            items.forEach(item => {
                subTotalPrice += Number(item.value);
            });

            subTotalPrice = parseFloat(subTotalPrice).toFixed(2);
            subTotalAmount = subTotalPrice;
            subTotalPrice = priceFormat(subTotalPrice);

            return subTotalPrice;
        }

        function priceFormat(subTotalPrice) {
            let parts = subTotalPrice.toString().split('.');
            subTotalPrice = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, ',') + (parts[1] ? '.' + parts[1] : '');

            return subTotalPrice;
        }

        function setPriceInfo() {
            let symbol = document.querySelector('.currency-symbol').value;

            // SubTotal
            let subTotalPrice = getSubTotalPrice();
            let subTotal = symbol + subTotalPrice;
            document.querySelector('.sub-total-amount').value = subTotalAmount;
            document.querySelector('.payment-table .sub-total').innerHTML = subTotal;

            // Discount
            let discountType = document.querySelector('#jform_discount_type').value;
            let discountValue = Number(document.querySelector('#jform_discount_value').value);
            if (discountValue > 0) {
                discountAmount = calculateDiscount(subTotalAmount, discountType, discountValue);
            } else {
                discountAmount = parseFloat(0.0).toFixed(2);
            }
            let discount = symbol + priceFormat(discountAmount);
            let discountOperator = '';
            if (discountAmount > 0) {
                discountOperator = '- ';
            }
            document.querySelector('.payment-table .discount').innerHTML = discountOperator + discount;

            // Shipping
            shippingAmount = Number(document.querySelector('.shipping-amount').value);
            document.querySelector('.payment-table .shipping').innerHTML =
                symbol + priceFormat(parseFloat(shippingAmount).toFixed(2));

            // Tax
            generateTaxAmount();
            taxAmount = Number(document.querySelector('.tax-amount').value);
            document.querySelector('.payment-table .tax').innerHTML =
                symbol + priceFormat(parseFloat(taxAmount).toFixed(2));

            // Coupon
            couponAmount = Number(document.querySelector('.coupon-amount').value);
            let couponOperator = '';
            if (couponAmount > 0) {
                couponOperator = '- ';
            }
            document.querySelector('.payment-table .coupon').innerHTML =
                couponOperator + symbol + priceFormat(parseFloat(couponAmount).toFixed(2));

            // TotalAmount
            totalAmount =
                Number(subTotalAmount) -
                Number(discountAmount) +
                Number(shippingAmount) +
                Number(taxAmount) -
                Number(couponAmount);
            totalAmount = parseFloat(totalAmount).toFixed(2);
            let total = symbol + priceFormat(totalAmount);

            document.querySelector('.payment-table .total').innerHTML = total;
        }

        function generateTaxAmount() {
            let items = document.querySelectorAll('.product-item');
            let saleTax = 0.0;
            items.forEach(item => {
                let quantity = Number(item.querySelector('.product-quantity').value);
                let taxAmount = Number(item.querySelector('.easystore-product-tax-amount').value);
                saleTax = parseFloat(Number(saleTax) + quantity * taxAmount).toFixed(2);
            });

            document.querySelector('.tax-amount').value = saleTax;
        }

        function calculateDiscount(price, type, discountValue) {
            let discountPrice;
            if (type === 'percent') {
                discountPrice = (Number(price) * Number(discountValue)) / 100;
            } else {
                discountPrice = Number(discountValue);
            }

            return parseFloat(discountPrice).toFixed(2);
        }

        function scrollToTop() {
            document.body.scrollTop = 0; // For Safari
            document.documentElement.scrollTop = 0; // For Chrome, Firefox, IE and Opera
        }
    });
})(Joomla, document);
