<?php

/**
 * @package     EasyStore.Site
 * @subpackage  EasyStore.Paddle
 *
 * @copyright   Copyright (C) 2023 JoomShaper <https://www.joomshaper.com>. All rights reserved.
 * @license     GNU General Public License version 3; see LICENSE
 */
namespace JoomShaper\Plugin\EasyStore\Paddle\Extension;

use Joomla\CMS\Factory;
use Joomla\Event\Event;
use Joomla\CMS\Language\Text;
use Joomla\Event\AbstractEvent;
use Joomla\CMS\Layout\LayoutHelper;
use JoomShaper\Plugin\EasyStore\Paddle\Utils\PaddleApi;
use JoomShaper\Plugin\EasyStore\Paddle\Utils\PaddleConstants;
use JoomShaper\Component\EasyStore\Site\Controller\CheckoutController;
use JoomShaper\Component\EasyStore\Administrator\Plugin\PaymentGatewayPlugin;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

class PaddlePayment extends PaymentGatewayPlugin {

    /**
     * Handles the payment process.
     * 
     * Gets the required data from the `PaddleConstants` class and creates a payment link.
     * Renders the checkout form and redirects the user to the payment link.
     * 
     * @param   AbstractEvent The event object that contains information about the payment. 
     * @since   1.0.0
     * @return  void
     */ 
    public function onPayment(Event $event){
        
        $paddleConstants = new PaddleConstants();
        $vendorID        = $paddleConstants->getVendorID();
        $vendorAuthKey   = $paddleConstants->getVendorAuthKey();

        if($vendorID && $vendorAuthKey){

            $arguments = $event->getArguments();      
            $products  = $arguments['subject'] ? $arguments['subject'] : '';

            if(!empty($products))
            {
                $paymentLink = PaddleApi::getPaymentLink($products,$arguments['order_id']);

                $layoutPath  = JPATH_ROOT . '/plugins/easystore/paddle/src/layouts';

                // Render the checkout form using the specified layout
                echo LayoutHelper::render('checkoutJs',['url' => $paymentLink, 'sellerID' => $vendorID], $layoutPath);

                $event->setArgument('redirectionUrl', $paymentLink);
            }
            
            
        }
        else {
           Factory::getApplication()->enqueueMessage(Text::_('PLG_EASYSTORE_PADDLE_NO_VENDOR_ID_AND_AUTH_KEY'),'error');
        }
    }

    /**
     * Handles payment notification from Paddle.
     *
     * Gets the required data from the request and verifies the signature.
     * Returns a 200 response code if the notification is valid.
     * 
     * @since 1.0.0
     */
    public function onPaymentNotify() {

        $errorReason       = null;
        $paddleConstants   = new PaddleConstants();
        // Paddle 'Public Key'
        $public_key_string = $paddleConstants->getPublicKey();
        $public_key        = openssl_get_publickey($public_key_string);

        
        // Get the p_signature parameter & base64 decode it.
        $signature = base64_decode($_POST['p_signature']);

        $eventType = $_POST['alert_name'];
        
        // Get the fields sent in the request, and remove the p_signature parameter
        $fields = $_POST;
        unset($fields['p_signature']);
        
        // ksort() and serialize the fields
        ksort($fields);
        foreach($fields as $k => $v) {
            if(!in_array(gettype($v), array('object', 'array'))) {
                $fields[$k] = "$v";
            }
        }
        
        $data = serialize($fields);
        
        // Verify the signature
        $verification = openssl_verify($data, $signature, $public_key, OPENSSL_ALGO_SHA1);

        if($verification == 1 && $eventType == 'payment_succeeded') {
            
            http_response_code(200);

            $passThrough   = json_decode($fields['passthrough']);
            $orderID       = $passThrough->orderID;
            $transactionId = $fields['order_id']; // Order id of paddle

            $checkoutController = new CheckoutController();
            $data               = (object) [
                'id'                   => $orderID,
                'payment_status'       => 'paid',
                'payment_error_reason' => $errorReason,
                'transaction_id'       => $transactionId
            ];

            try {
                $checkoutController->updateOrder($data);               
                $checkoutController->onOrderPlacementCompletion() ;              
            } catch (\Throwable $error) {
                Factory::getApplication()->enqueueMessage($error->getMessage(), 'error');
            }
        }
    }
}
?>