<?php

/**
 * @package     EasyStore.Site
 * @subpackage  EasyStore.paypal
 *
 * @copyright   Copyright (C) 2023 - 2024 JoomShaper <https://www.joomshaper.com>. All rights reserved.
 * @license     GNU General Public License version 3; see LICENSE
 */
namespace JoomShaper\Plugin\EasyStore\Paypal\Extension;


use Exception;
use Joomla\CMS\Log\Log;
use Joomla\Event\Event;
use Joomla\CMS\Http\HttpFactory;
use Joomla\CMS\Application\CMSApplication;
use JoomShaper\Plugin\EasyStore\Paypal\Utils\PaypalConstants;
use JoomShaper\Component\EasyStore\Administrator\Plugin\PaymentGatewayPlugin;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects


class PaypalPayment extends PaymentGatewayPlugin {

    /** @var CMSApplication */
    protected $app;
    
    /**
     * Handle the payment process for the event.
     * Redirects the user to PayPal for payment processing.
     *
     * @param Event $event The payment event containing necessary information.
     * @since 1.0.0
     */
    public function onPayment(Event $event){
        
        $constant     = new PaypalConstants();
        $paypalUrl    = $constant->getPaypalUrl();
        $arguments    = $event->getArguments();      
        $paymentData  = $arguments['subject'] ?: new \stdClass();

        $query = [];
        $query['cmd']           = '_cart';
        $query['upload']        = '1';
        $query['business']      = $constant->getMerchantEmail();
        $query['custom']        = json_encode(['order_id' => $paymentData->order_id]);       
        $query['notify_url']    = $constant->getWebHookUrl();
        $query['return']        = $constant->getSuccessUrl($paymentData->order_id);
        $query['cancel_return'] = $constant->getCancelUrl($paymentData->order_id);
        
        if (!empty($paymentData)) {

            if(!empty($paymentData->items))
            {
                foreach ($paymentData->items as $key => $product) {
                    
                    $price = $product->discounted_price ?: $product->regular_price;
                    $index = $key + 1;

                    $query["item_name_{$index}"]   = $product->title;
                    $query["item_number_{$index}"] = (string)$product->id;
                    $query["quantity_{$index}"]    = (string)$product->quantity;
                    $query["amount_{$index}"]      = (string)$price;                  
                }
            }

            $query['tax_cart']             = (string)$paymentData->tax;
            $query['currency_code']        = strtoupper($paymentData->currency);
            $query['handling_cart']        = $paymentData->shipping_charge;  
            $query['discount_amount_cart'] = (string)$paymentData->coupon_discount_amount;     
        }
        
        $queryString = http_build_query($query);     
        $IPNUrl      = $paypalUrl . '?' . $queryString;

        try {
            $this->app->redirect($IPNUrl);
        } catch (\Throwable $error) {
            Log::add($error->getMessage(), Log::ERROR, 'paypal.easystore');
            $this->app->enqueueMessage($error->getMessage(), 'error');
            $this->app->redirect($paymentData->back_to_checkout_page);
        }
    }

    /**
     * Process PayPal IPN Notification.
     *
     * This function handles the verification of PayPal's Instant Payment Notification (IPN).
     * It receives the IPN data, constructs a verification request, and sends it back to PayPal using cURL.
     * The response from PayPal is then checked to determine if the IPN is verified or invalid.
     *
     * @return void
     * @throws Exception If cURL encounters an error or if PayPal's response is not as expected.
     */
    public function onPaymentNotify(Event $event)
    {
        // Event Arguments
        $arguments         = $event->getArguments();
        $paymentNotifyData = $arguments['subject'] ?: new \stdClass();

        $constant        = new PaypalConstants();
        $input           = $this->app->input;
        $rawPayload      = $paymentNotifyData->raw_payload;
        $rawPayloadArray = explode('&', $rawPayload);
        $order           = $paymentNotifyData->order;
        $callBackData    = [];
        $reason          = null;

        http_response_code(200);
     
        foreach ($rawPayloadArray as $keyValue) {
            $keyValue = explode('=', $keyValue);

            if (count($keyValue) === 2) {
                // Since we do not want the plus in the datetime string to be encoded to a space, we manually encode it.
                if ($keyValue[0] === 'payment_date') {
                    if (substr_count($keyValue[1], '+') === 1) {
                        $keyValue[1] = str_replace('+', '%2B', $keyValue[1]);
                    }
                }
                $callBackData[$keyValue[0]] = urldecode($keyValue[1]);
            }
        }

        // Build the body of the verification post request, adding the _notify-validate command.
        $requestData             = 'cmd=_notify-validate';
        $get_magic_quotes_exists = false;

        if (function_exists('get_magic_quotes_gpc')) {
            $get_magic_quotes_exists = true;
        }

        foreach ($callBackData as $key => $value) {
            if ($get_magic_quotes_exists) {
                $value = urlencode(stripslashes($value));
            } else {
                $value = urlencode($value);
            }

            $requestData .= "&$key=$value";
        }

        // Post the data back to PayPal, using curl. Throw exceptions if errors occur.
        $paypalUrl            = $constant->getPaypalUrl();
        $merchantEmail        = $constant->getMerchantEmail();
        $merchantEmailFromIPN = $input->get('receiver_email','','RAW');

        // Create an HTTP client
        $headers = [
            'User-Agent: PHP-IPN-Verification-Script',
            'Connection: Close',
        ];
        
        $response        = HttpFactory::getHttp()->post($paypalUrl,$requestData,$headers,30);
        $responseMessage = (string) $response->getBody();
        $http_code       = $response->getStatusCode();
        
        if ($http_code !== 200) {
            Log::add("PayPal responded with http code $http_code", Log::ERROR, 'paypal.easystore');
            throw new Exception("PayPal responded with http code $http_code");
        }

        // Check if PayPal verifies the IPN data, and if so, return true.
        if (strcmp ($responseMessage, "VERIFIED") === 0) {
            if($input->get('txn_id') && $merchantEmail === $merchantEmailFromIPN)
            {
                $orderID       = json_decode($input->get('custom','','STRING'));
                $transactionID = $input->get('txn_id');

                $paymentStatus = $input->get('payment_status'); 

                switch ($paymentStatus) 
                {
                    case 'Pending':
                        $reason = $input->get('pending_reason','','STRING');                       
                        break;
                    case 'Denied':
                        $reason = $input->get('reason_code');
                        break;
                    case 'Completed':
                        $paymentStatus = 'paid';
                }

                $data = (object) [
                    'id'                   => $orderID->order_id,
                    'payment_status'       => strtolower($paymentStatus),
                    'payment_error_reason' => $reason,
                    'transaction_id'       => $transactionID
                ];

                try {
                    $order->updateOrder($data);
                    
                    if ($paymentStatus === 'paid'){
                        $order->onOrderPlacementCompletion();
                    }
                    
                } catch (\Throwable $error) {
                    Log::add($error->getMessage(),Log::ERROR, 'paypal.easystore');
                    $this->app->enqueueMessage($error->getMessage(), 'error');
                }
                
            }

            
        } 
        else if (strcmp($responseMessage, "INVALID") == 0) {
            // IPN invalid, log for manual investigation
            Log::add("The response from IPN was: <b>" . $responseMessage() ."</b>", Log::ERROR, 'paypal.easystore');
            return false;
        }
    }


    /**
     * Check if all the required fields for the plugin are filled.
     *
     * @return void The result of the check, indicating whether the required fields are filled.
     * @since 1.0.3
     */
    public function onBeforePayment(Event $event)
    {
        $isRequiredFieldsFilled = !empty((new PaypalConstants())->getMerchantEmail());
        $event->setArgument('result', $isRequiredFieldsFilled);
    }
}
?>