<?php

/**
 * @package     EasyStore.Site
 * @subpackage  EasyStore.stripe
 *
 * @copyright   Copyright (C) 2023 JoomShaper <https://www.joomshaper.com>. All rights reserved.
 * @license     GNU General Public License version 3; see LICENSE
 */
namespace JoomShaper\Plugin\EasyStore\Stripe\Extension;

require_once __DIR__ . '/../../vendor/autoload.php';

use Stripe\Stripe;
use Joomla\CMS\Factory;
use Joomla\Event\Event;
use Stripe\StripeClient;
use Stripe\Exception\ApiErrorException;
use JoomShaper\Plugin\EasyStore\Stripe\Utils\StripeApi;
use JoomShaper\Plugin\EasyStore\Stripe\Utils\StripeConstants;
use JoomShaper\Component\EasyStore\Site\Controller\CheckoutController;
use JoomShaper\Component\EasyStore\Administrator\Plugin\PaymentGatewayPlugin;

/**
 * StripePayment class.
 * 
 * This class extends the PaymentGatewayPlugin class and provides implementation for integrating the Stripe 
 * payment gateway.
 * 
 * It handles various payment-related functionalities such as initiating payments, handling successful 
 * payments and managing canceled payments.
 *
 * @since 1.0.0
 */
class StripePayment extends PaymentGatewayPlugin {
    
    /**
     * Initiate an event that will lead to a redirection to the Stripe checkout page.
     *
     * @param Event|string  $event  The event object.
     * @since 1.0.0
     */
    public function onPayment(Event $event){

        $stripeConstants = new StripeConstants();
        $stripeSecretKey = $stripeConstants->getStripeSecretKey();

        if (empty($stripeSecretKey)) {
            return false;
        }

        Stripe::setApiKey($stripeSecretKey);
        header('Content-Type: application/json');

        $lineItems   = [];
        $productInfo = [];

        $eventArguments = $event->getArguments();      
        $productsList   = $eventArguments['subject'] ? $eventArguments['subject'] : [];

        if (!empty($productsList)) {
            //Create Line items for products
            foreach ($productsList->items as $product) {               
                $price = $product->discounted_price ? $product->discounted_price : $product->regular_price;
                
                $productInfo =  [           
                    'price_data' => [                  
                        'product_data' => [
                            'name'          => $product->title,                    
                        ],
                        'unit_amount' => (float)$price * 100,
                        'currency'    => strtolower($productsList->currency), 
                    ],
                    'quantity' => $product->quantity,                   
                ];
                $lineItems [] = $productInfo;
            }

            // Create a line item for tax
            if ($productsList->tax) {
                $taxLineItem = [
                    'price_data' => [
                        'product_data' => [
                            'name' => 'Tax',
                        ],
                        'unit_amount' => $productsList->tax * 100, 
                        'currency' => $productsList->currency, 
                    ],
                    'quantity' => 1,
                ];
                $lineItems[] = $taxLineItem;
            }           
        }   

        try
        {           
            $checkoutUrl = StripeApi::getCheckoutUrl($lineItems,$eventArguments['order_id'],$productsList);
            // Redirect to checkout page
            $event->setArgument('redirectionUrl', $checkoutUrl);
        }
        catch (ApiErrorException $e)
        {
            Factory::getApplication()->enqueueMessage($e->getMessage(), 'error');
        }
    }

    /**
     * Handle payment notifications from Stripe webhook.
     * 
     * @since 1.0.0
     */
    public function onPaymentNotify(){
        
        $errorReason     = null;
        $stripeConstant  = new StripeConstants();
        $stripeSecretKey = $stripeConstant->getStripeSecretKey();

        Stripe::setApiKey($stripeSecretKey);

        $endpoint_secret = $stripeConstant->getWebHookSecret();

        $payload = @file_get_contents('php://input');
        $event   = null;       

        try {
            $event = \Stripe\Event::constructFrom(json_decode($payload, true));
        } catch(\UnexpectedValueException $e) {
            // Invalid payload
            echo '⚠️  Webhook error while parsing basic request.';
            http_response_code(400);
            exit();
        }
        if ($endpoint_secret) {
            $sig_header = $_SERVER['HTTP_STRIPE_SIGNATURE'];
            try {
                $event = \Stripe\Webhook::constructEvent(
                    $payload, $sig_header, $endpoint_secret
                );
                http_response_code(200);
                
            } catch(\Stripe\Exception\SignatureVerificationException $e) {
                // Invalid signature
                echo '⚠️  Webhook error while validating signature.';
                http_response_code(400);
                exit();
            }
        }

        $paymentDetails = $event->data->object; 
        $orderID        = $paymentDetails->metadata->order_id;
        $transactionId  = $paymentDetails->payment_intent;
        $couponDiscount = isset($paymentDetails->total_details) ? $paymentDetails->total_details->amount_discount : 0;
        $coupon_id      = $paymentDetails->metadata->coupon_id;

        // Handle the event
       if ($event->type === 'payment_intent.payment_failed') {
            $errorReason = $paymentDetails->last_payment_error->message;
            $status      = 'failed';
        } elseif ($event->type === 'checkout.session.completed') {
            $status = 'paid';
        }
        elseif ($event->type === 'payment_intent.canceled') {
            $status = 'cancelled';
        }

        // Delete coupon if discount is applied
        if ($couponDiscount > 0 && $coupon_id) {
            StripeApi::deleteCoupon($coupon_id);
        }

        $checkoutController = new CheckoutController();
        $data               = (object) [
            'id'                   => $orderID,
            'payment_status'       => $status,
            'payment_error_reason' => $errorReason,
            'transaction_id'       => $transactionId
        ];

        try {
            $checkoutController->updateOrder($data);
            
            if ($status === 'paid'){
                $checkoutController->deductQuantityFromInventory();
            }
            
        } catch (ApiErrorException $error) {
            Factory::getApplication()->enqueueMessage($error->getMessage(), 'error');
        }

        exit();
    }


    /**
     * Check if a webhook URL exists in Stripe, and create a new one if it doesn't.
     *
     * @return string|bool The webhook secret if a new webhook is created, or false if the URL already exists.
     * @since  1.0.0
     */
    public function onCreateWebhook()
    {
        $webhookUrlIsAdded = false;
        
        $stripeConstant = new StripeConstants();
        $webhookUrl     = $stripeConstant->getWebHookUrl();

        $stripe      = new StripeClient($stripeConstant->getStripeSecretKey());
        $allWebhooks = $stripe->webhookEndpoints->all();

        foreach ($allWebhooks as $webhook) {
            if ($webhook->url === $webhookUrl) {
                $webhookUrlIsAdded = true;
                break;
            }
        }

        if(!$webhookUrlIsAdded){
            $newWebhook = StripeApi::createNewWebhookUrl($stripeConstant->getStripeSecretKey());
            
            $webHookSecret = ($newWebhook->secret) ?: null;
            return $webHookSecret;
        }
        
        return false;
    }

}
